package mezz.jei.fabric.platform;

import com.mojang.serialization.Codec;
import com.mojang.serialization.DataResult;
import com.mojang.serialization.codecs.RecordCodecBuilder;
import mezz.jei.api.fabric.constants.FabricTypes;
import mezz.jei.api.fabric.ingredients.fluids.IJeiFluidIngredient;
import mezz.jei.api.ingredients.IIngredientRenderer;
import mezz.jei.api.ingredients.IIngredientTypeWithSubtypes;
import mezz.jei.api.ingredients.ITypedIngredient;
import mezz.jei.common.platform.IPlatformFluidHelperInternal;
import mezz.jei.fabric.ingredients.fluid.JeiFluidIngredient;
import mezz.jei.library.render.FluidTankRenderer;
import net.fabricmc.fabric.api.transfer.v1.client.fluid.FluidVariantRendering;
import net.fabricmc.fabric.api.transfer.v1.context.ContainerItemContext;
import net.fabricmc.fabric.api.transfer.v1.fluid.FluidConstants;
import net.fabricmc.fabric.api.transfer.v1.fluid.FluidStorage;
import net.fabricmc.fabric.api.transfer.v1.fluid.FluidVariant;
import net.fabricmc.fabric.api.transfer.v1.fluid.FluidVariantAttributes;
import net.fabricmc.fabric.api.transfer.v1.storage.Storage;
import net.minecraft.class_1058;
import net.minecraft.class_1657;
import net.minecraft.class_1792;
import net.minecraft.class_1836;
import net.minecraft.class_2561;
import net.minecraft.class_3611;
import net.minecraft.class_6880;
import net.minecraft.class_9326;
import org.jetbrains.annotations.Nullable;

import java.util.Iterator;
import java.util.List;
import java.util.Optional;

public class FluidHelper implements IPlatformFluidHelperInternal<IJeiFluidIngredient> {
	private static final Codec<Long> POSITIVE_LONG = Codec.LONG.validate((integer) -> {
		if (integer.compareTo((long) 1) >= 0 && integer.compareTo(Long.MAX_VALUE) <= 0) {
			return DataResult.success(integer);
		}
		return DataResult.error(() -> "Value must be positive: " + integer);
	});

	private static final Codec<IJeiFluidIngredient> CODEC = Codec.lazyInitialized(() -> {
		return RecordCodecBuilder.create((builder) -> {
			return builder.group(
					FluidVariant.CODEC.fieldOf("variant")
						.forGetter(IJeiFluidIngredient::getFluidVariant),
					POSITIVE_LONG.fieldOf("amount")
						.forGetter(IJeiFluidIngredient::getAmount)
				)
				.apply(builder, JeiFluidIngredient::new);
		});
	});

	private static final Codec<IJeiFluidIngredient> NORMALIZED_CODEC = Codec.lazyInitialized(() -> {
		return FluidVariant.CODEC.xmap(
			fluidVariant -> {
				return new JeiFluidIngredient(fluidVariant, FluidConstants.BUCKET);
			},
			IJeiFluidIngredient::getFluidVariant
		);
	});

	@Override
	public IIngredientTypeWithSubtypes<class_3611, IJeiFluidIngredient> getFluidIngredientType() {
		return FabricTypes.FLUID_STACK;
	}

	@Override
	public IIngredientRenderer<IJeiFluidIngredient> createRenderer(long capacity, boolean showCapacity, int width, int height) {
		return new FluidTankRenderer<>(this, capacity, showCapacity, width, height);
	}

	@Override
	public Optional<class_1058> getStillFluidSprite(IJeiFluidIngredient ingredient) {
		FluidVariant fluidVariant = ingredient.getFluidVariant();
		class_1058 sprite = FluidVariantRendering.getSprite(fluidVariant);
		return Optional.ofNullable(sprite);
	}

	@Override
	public class_2561 getDisplayName(IJeiFluidIngredient ingredient) {
		FluidVariant fluidVariant = ingredient.getFluidVariant();
		class_2561 displayName = FluidVariantAttributes.getName(fluidVariant);

		class_3611 fluid = ingredient.getFluidVariant().getFluid();
		if (!fluid.method_15793(fluid.method_15785())) {
			return class_2561.method_43469("jei.tooltip.liquid.flowing", displayName);
		}
		return displayName;
	}

	@Override
	public int getColorTint(IJeiFluidIngredient ingredient) {
		FluidVariant fluidVariant = ingredient.getFluidVariant();
		int fluidColor = FluidVariantRendering.getColor(fluidVariant);
		return fluidColor | 0xFF000000;
	}

	@Override
	public List<class_2561> getTooltip(IJeiFluidIngredient ingredient, @Nullable class_1657 player, class_1792.class_9635 tooltipContext, class_1836 tooltipFlag) {
		FluidVariant fluidVariant = ingredient.getFluidVariant();
		return FluidVariantRendering.getTooltip(fluidVariant, tooltipFlag);
	}

	@Override
	public long getAmount(IJeiFluidIngredient ingredient) {
		return ingredient.getAmount();
	}

	@Override
	public class_9326 getComponentsPatch(IJeiFluidIngredient ingredient) {
		FluidVariant fluid = ingredient.getFluidVariant();
		return fluid.getComponents();
	}

	@Override
	public long bucketVolume() {
		return FluidConstants.BUCKET;
	}

	@Override
	public IJeiFluidIngredient create(class_6880<class_3611> fluid, long amount, class_9326 patch) {
		FluidVariant fluidVariant = FluidVariant.of(fluid.comp_349(), patch);
		return new JeiFluidIngredient(fluidVariant, amount);
	}

	@Override
	public IJeiFluidIngredient create(class_6880<class_3611> fluid, long amount) {
		FluidVariant fluidVariant = FluidVariant.of(fluid.comp_349());
		return new JeiFluidIngredient(fluidVariant, amount);
	}

	@Override
	public IJeiFluidIngredient copy(IJeiFluidIngredient ingredient) {
		return new JeiFluidIngredient(ingredient.getFluidVariant(), ingredient.getAmount());
	}

	@Override
	public IJeiFluidIngredient normalize(IJeiFluidIngredient ingredient) {
		if (ingredient.getAmount() == bucketVolume()) {
			return ingredient;
		}
		return new JeiFluidIngredient(ingredient.getFluidVariant(), bucketVolume());
	}

	@Override
	public Optional<IJeiFluidIngredient> getContainedFluid(ITypedIngredient<?> ingredient) {
		return ingredient.getItemStack()
			.map(ContainerItemContext::withConstant)
			.map(c -> c.find(FluidStorage.ITEM))
			.map(Storage::iterator)
			.filter(Iterator::hasNext)
			.map(Iterator::next)
			.map(view -> {
				FluidVariant resource = view.getResource();
				return new JeiFluidIngredient(resource, view.getAmount());
			});
	}

	@Override
	public IJeiFluidIngredient copyWithAmount(IJeiFluidIngredient ingredient, long amount) {
		return new JeiFluidIngredient(ingredient.getFluidVariant(), amount);
	}

	@Override
	public Codec<IJeiFluidIngredient> getCodec() {
		return Codec.withAlternative(
			NORMALIZED_CODEC,
			CODEC // TODO: remove this fallback codec in the next major version of JEI
		);
	}
}
