package mezz.jei.api.recipe.types;

import mezz.jei.api.constants.RecipeTypes;
import mezz.jei.api.recipe.category.IRecipeCategory;
import net.minecraft.class_1860;
import net.minecraft.class_2960;
import net.minecraft.class_3956;
import net.minecraft.class_8786;

/**
 * Identifies a type of recipe, (i.e. Crafting Table Recipe, Furnace Recipe).
 * Each {@link IRecipeCategory} can be uniquely identified by its {@link IRecipeType}.
 *
 * Unfortunately, the vanilla {@link class_3956} only works for recipes that extend the vanilla {@link class_1860} class,
 * so this more general version is needed for modded recipes in JEI.
 *
 * @see RecipeTypes for all the built-in recipe types that are added by JEI.
 * @see IRecipeHolderType for a convenient {@link IRecipeType} created from a vanilla {@link class_3956}
 *
 * @since 20.0.0
 *
 * @apiNote Replaces RecipeType in 20.0.0 to avoid naming collision with the vanilla {@link class_3956}
 */
public interface IRecipeType<T> {
	/**
	 * The unique id of this recipe type.
	 *
	 * @since 20.0.0
	 */
	class_2960 getUid();

	/**
	 * The class of recipes represented by this recipe type.
	 *
	 * @since 20.0.0
	 */
	Class<? extends T> getRecipeClass();

	/**
	 * Create a JEI RecipeType from a Vanilla RecipeType.
	 * Returns a RecipeType that uses {@link class_8786} to hold recipes.
	 * @since 20.0.0
	 */
	static <R extends class_1860<?>> IRecipeHolderType<R> create(class_3956<R> vanillaRecipeType) {
		return IRecipeHolderType.create(vanillaRecipeType);
	}

	/**
	 * Create a JEI RecipeType from a given uid.
	 * Returns a RecipeType that uses the given recipe class to hold recipes.
	 * @since 20.0.0
	 */
	static <T> IRecipeType<T> create(class_2960 uid, Class<? extends T> recipeClass) {
		return new JeiRecipeType<>(uid, recipeClass);
	}

	/**
	 * Convenience function create a JEI RecipeType from a given nameSpace and path.
	 * Returns a RecipeType that uses the given recipe class to hold recipes.
	 * @since 20.0.0
	 */
	static <T> IRecipeType<T> create(String nameSpace, String path, Class<? extends T> recipeClass) {
		class_2960 uid = class_2960.method_60655(nameSpace, path);
		return create(uid, recipeClass);
	}

	record JeiRecipeType<T>(class_2960 uid, Class<? extends T> recipeClass) implements IRecipeType<T> {
		@Override
		public class_2960 getUid() {
			return uid;
		}

		@Override
		public Class<? extends T> getRecipeClass() {
			return recipeClass;
		}
	}
}
