package mezz.jei.library.recipes;

import mezz.jei.api.recipe.category.extensions.vanilla.crafting.ICraftingCategoryExtension;
import mezz.jei.library.util.RecipeErrorUtil;
import net.minecraft.class_1860;
import net.minecraft.class_3955;
import net.minecraft.class_8786;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.jetbrains.annotations.Nullable;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.IdentityHashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Stream;

public class CraftingExtensionHelper {
	private static final Logger LOGGER = LogManager.getLogger();

	private final List<Handler<? extends class_3955>> handlers = new ArrayList<>();
	private final Set<Class<? extends class_3955>> handledClasses = new HashSet<>();
	private final Map<class_8786<? extends class_3955>, @Nullable ICraftingCategoryExtension<? extends class_3955>> cache = new IdentityHashMap<>();

	public <T extends class_3955> void addRecipeExtension(Class<? extends T> recipeClass, ICraftingCategoryExtension<T> recipeExtension) {
		if (!class_3955.class.isAssignableFrom(recipeClass)) {
			throw new IllegalArgumentException("Recipe handlers must handle a specific class that inherits from CraftingRecipe. Instead got: " + recipeClass);
		}
		if (this.handledClasses.contains(recipeClass)) {
			throw new IllegalArgumentException("A Recipe Extension has already been registered for this class:" + recipeClass);
		}
		this.handledClasses.add(recipeClass);
		this.handlers.add(new Handler<>(recipeClass, recipeExtension));
	}

	public <R extends class_3955> ICraftingCategoryExtension<R> getRecipeExtension(class_8786<R> recipeHolder) {
		return getOptionalRecipeExtension(recipeHolder)
			.orElseThrow(() -> {
				String recipeName = RecipeErrorUtil.getNameForRecipe(recipeHolder);
				return new RuntimeException("Failed to create recipe extension for recipe: " + recipeName);
			});
	}

	public <R extends class_3955> Optional<ICraftingCategoryExtension<R>> getOptionalRecipeExtension(class_8786<R> recipeHolder) {
		if (cache.containsKey(recipeHolder)) {
			ICraftingCategoryExtension<? extends class_3955> extension = cache.get(recipeHolder);
			if (extension != null) {
				@SuppressWarnings("unchecked")
				ICraftingCategoryExtension<R> cast = (ICraftingCategoryExtension<R>) extension;
				return Optional.of(cast);
			}
			return Optional.empty();
		}

		Optional<ICraftingCategoryExtension<R>> result = getBestRecipeHandler(recipeHolder)
			.map(Handler::getExtension);

		cache.put(recipeHolder, result.orElse(null));

		return result;
	}

	private <T extends class_3955> Stream<Handler<T>> getRecipeHandlerStream(class_8786<T> recipeHolder) {
		return handlers.stream()
			.flatMap(handler -> handler.optionalCast(recipeHolder).stream());
	}

	private <T extends class_3955> Optional<Handler<T>> getBestRecipeHandler(class_8786<T> recipeHolder) {
		Class<? extends class_3955> recipeClass = recipeHolder.comp_1933().getClass();

		List<Handler<T>> assignableHandlers = new ArrayList<>();
		// try to find an exact match
		List<Handler<T>> allHandlers = getRecipeHandlerStream(recipeHolder).toList();
		for (Handler<T> handler : allHandlers) {
			Class<? extends class_3955> handlerRecipeClass = handler.getRecipeClass();
			if (handlerRecipeClass.equals(recipeClass)) {
				return Optional.of(handler);
			}
			// remove any handlers that are super of this one
			assignableHandlers.removeIf(h -> h.getRecipeClass().isAssignableFrom(handlerRecipeClass));
			// only add this if it's not a super class of another assignable handler
			if (assignableHandlers.stream().noneMatch(h -> handlerRecipeClass.isAssignableFrom(h.getRecipeClass()))) {
				assignableHandlers.add(handler);
			}
		}
		if (assignableHandlers.isEmpty()) {
			return Optional.empty();
		}
		if (assignableHandlers.size() == 1) {
			return Optional.of(assignableHandlers.getFirst());
		}

		// try super classes to get the closest match
		Class<?> superClass = recipeClass;
		while (!Object.class.equals(superClass)) {
			superClass = superClass.getSuperclass();
			for (Handler<T> handler : assignableHandlers) {
				if (handler.getRecipeClass().equals(superClass)) {
					return Optional.of(handler);
				}
			}
		}

		List<Class<? extends class_3955>> assignableClasses = assignableHandlers.stream()
			.<Class<? extends class_3955>>map(Handler::getRecipeClass)
			.toList();
		LOGGER.warn("Found multiple matching recipe handlers for {}: {}", recipeClass, assignableClasses);
		return Optional.of(assignableHandlers.getFirst());
	}

	private record Handler<T extends class_3955>(
		Class<? extends T> recipeClass,
		ICraftingCategoryExtension<T> extension
	) {
		public <V extends class_3955> Optional<Handler<V>> optionalCast(class_8786<V> recipeHolder) {
			if (isHandled(recipeHolder)) {
				@SuppressWarnings("unchecked")
				Handler<V> cast = (Handler<V>) this;
				return Optional.of(cast);
			}
			return Optional.empty();
		}

		public boolean isHandled(class_8786<?> recipeHolder) {
			class_1860<?> recipe = recipeHolder.comp_1933();
			if (recipeClass.isInstance(recipe)) {
				@SuppressWarnings("unchecked")
				class_8786<T> cast = (class_8786<T>) recipeHolder;
				return extension.isHandled(cast);
			}
			return false;
		}

		public Class<? extends T> getRecipeClass() {
			return recipeClass;
		}

		public ICraftingCategoryExtension<T> getExtension() {
			return extension;
		}
	}

}
