package mezz.jei.common.util;

import mezz.jei.api.ingredients.IIngredientHelper;
import mezz.jei.api.ingredients.IIngredientType;
import mezz.jei.api.ingredients.subtypes.UidContext;
import mezz.jei.api.recipe.RecipeType;
import mezz.jei.api.runtime.IIngredientManager;
import mezz.jei.common.platform.IPlatformModHelper;
import mezz.jei.common.platform.IPlatformRegistry;
import mezz.jei.common.platform.Services;
import net.minecraft.class_128;
import net.minecraft.class_129;
import net.minecraft.class_1747;
import net.minecraft.class_1792;
import net.minecraft.class_1799;
import net.minecraft.class_2248;
import net.minecraft.class_2371;
import net.minecraft.class_2378;
import net.minecraft.class_2487;
import net.minecraft.class_2960;
import net.minecraft.class_310;
import net.minecraft.class_7924;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.jetbrains.annotations.Nullable;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

public final class ErrorUtil {
	private static final Logger LOGGER = LogManager.getLogger();

	private ErrorUtil() {
	}

	public static <T> String getIngredientInfo(T ingredient, IIngredientType<T> ingredientType, IIngredientManager ingredientManager) {
		IIngredientHelper<T> ingredientHelper = ingredientManager.getIngredientHelper(ingredientType);
		return ingredientHelper.getErrorInfo(ingredient);
	}

	public static String getItemStackInfo(@Nullable class_1799 itemStack) {
		if (itemStack == null) {
			return "null";
		}
		class_1792 item = itemStack.method_7909();
		IPlatformRegistry<class_1792> itemRegistry = Services.PLATFORM.getRegistry(class_7924.field_41197);

		final String itemName = itemRegistry.getRegistryName(item)
			.map(class_2960::toString)
			.orElseGet(() -> {
				if (item instanceof class_1747 blockItem) {
					final String blockName = getBlockName(blockItem);
					return "BlockItem(" + blockName + ")";
				} else {
					return item.getClass().getName();
				}
			});

		class_2487 nbt = itemStack.method_7969();
		if (nbt != null) {
			return itemStack + " " + itemName + " nbt:" + nbt;
		}
		return itemStack + " " + itemName;
	}

	@SuppressWarnings("ConstantValue")
	private static String getBlockName(class_1747 blockItem) {
		class_2248 block = blockItem.method_7711();
		if (block == null) {
			return "null";
		}
		class_2378<class_2248> blockRegistry = RegistryUtil.getRegistry(class_7924.field_41254);
		class_2960 key = blockRegistry.method_10221(block);
		if (key != null) {
			return key.toString();
		}
		return block.getClass().getName();
	}


	@SuppressWarnings("ConstantConditions")
	public static void checkNotEmpty(class_1799 itemStack) {
		if (itemStack == null) {
			throw new NullPointerException("ItemStack must not be null.");
		} else if (itemStack.method_7960()) {
			String info = getItemStackInfo(itemStack);
			throw new IllegalArgumentException("ItemStack value must not be empty. " + info);
		}
	}

	@SuppressWarnings("ConstantConditions")
	public static void checkNotEmpty(class_1799 itemStack, String name) {
		if (itemStack == null) {
			throw new NullPointerException(name + " must not be null.");
		} else if (itemStack.method_7960()) {
			String info = getItemStackInfo(itemStack);
			throw new IllegalArgumentException("ItemStack " + name + " must not be empty. " + info);
		}
	}

	@SuppressWarnings("ConstantConditions")
	public static <T> void checkNotEmpty(T[] values, String name) {
		if (values == null) {
			throw new NullPointerException(name + " must not be null.");
		} else if (values.length == 0) {
			throw new IllegalArgumentException(name + " must not be empty.");
		}
		for (T value : values) {
			if (value == null) {
				throw new NullPointerException(name + " must not contain null values.");
			}
		}
	}

	@SuppressWarnings("ConstantConditions")
	public static void checkNotEmpty(Collection<?> values, String name) {
		if (values == null) {
			throw new NullPointerException(name + " must not be null.");
		} else if (values.isEmpty()) {
			throw new IllegalArgumentException(name + " must not be empty.");
		} else if (!(values instanceof class_2371)) {
			for (Object value : values) {
				if (value == null) {
					throw new NullPointerException(name + " must not contain null values.");
				}
			}
		}
	}

	public static <T> void checkNotNull(@Nullable T object, String name) {
		if (object == null) {
			throw new NullPointerException(name + " must not be null.");
		}
	}

	@SuppressWarnings("ConstantConditions")
	public static void checkNotNull(Collection<?> values, String name) {
		if (values == null) {
			throw new NullPointerException(name + " must not be null.");
		} else if (!(values instanceof class_2371)) {
			for (Object value : values) {
				if (value == null) {
					throw new NullPointerException(name + " must not contain null values.");
				}
			}
		}
	}

	@SuppressWarnings("ConstantConditions")
	public static void assertMainThread() {
		class_310 minecraft = class_310.method_1551();
		if (minecraft != null && !minecraft.method_18854()) {
			Thread currentThread = Thread.currentThread();
			throw new IllegalStateException(
				"A JEI API method is being called by another mod from the wrong thread:\n" +
					currentThread + "\n" +
					"It must be called on the main thread by using Minecraft.addScheduledTask."
			);
		}
	}

	public static <T> void validateRecipes(RecipeType<T> recipeType, Iterable<? extends T> recipes) {
		Class<?> recipeClass = recipeType.getRecipeClass();
		for (T recipe : recipes) {
			if (!recipeClass.isInstance(recipe)) {
				throw new IllegalArgumentException(recipeType + " recipes must be an instance of " + recipeClass + ". Instead got: " + recipe.getClass());
			}
		}
	}

	public static <T> class_128 createIngredientCrashReport(Throwable throwable, String title, IIngredientManager ingredientManager, IIngredientType<T> ingredientType, T ingredient) {
		class_128 crashReport = class_128.method_560(throwable, title);
		class_129 category = crashReport.method_562("Ingredient");
		setIngredientCategoryDetails(category, ingredientType, ingredient, ingredientManager);
		return crashReport;
	}

	public static <T> void logIngredientCrash(Throwable throwable, String title, IIngredientManager ingredientManager, IIngredientType<T> ingredientType, T ingredient) {
		class_129 category = new class_129("Ingredient");
		setIngredientCategoryDetails(category, ingredientType, ingredient, ingredientManager);
		LOGGER.error(crashReportToString(throwable, title, List.of(category)));
	}

	public static <T> void logIngredientCrash(
		Throwable throwable,
		String title,
		IIngredientManager ingredientManager,
		IIngredientType<T> ingredientType,
		T ingredient,
		class_129... extraCategories
	) {
		class_129 category = new class_129("Ingredient");
		setIngredientCategoryDetails(category, ingredientType, ingredient, ingredientManager);
		List<class_129> categoryList = new ArrayList<>();
		categoryList.add(category);
		categoryList.addAll(List.of(extraCategories));
		LOGGER.error(crashReportToString(throwable, title, categoryList));
	}

	private static <T> void setIngredientCategoryDetails(class_129 category, IIngredientType<T> ingredientType, T ingredient, IIngredientManager ingredientManager) {
		IIngredientHelper<T> ingredientHelper = ingredientManager.getIngredientHelper(ingredientType);
		IPlatformModHelper modHelper = Services.PLATFORM.getModHelper();

		category.method_577("Name", () -> ingredientHelper.getDisplayName(ingredient));
		category.method_577("Mod's Name", () -> {
			String modId = ingredientHelper.getDisplayModId(ingredient);
			return modHelper.getModNameForModId(modId);
		});
		category.method_577("Registry Name", () -> ingredientHelper.getResourceLocation(ingredient).toString());
		category.method_577("Class Name", () -> ingredient.getClass().toString());
		category.method_577("toString Name", ingredient::toString);
		category.method_577("Unique Id for JEI (for JEI Blacklist)", () -> ingredientHelper.getUniqueId(ingredient, UidContext.Ingredient));
		category.method_577("Ingredient Type for JEI", () -> ingredientType.getIngredientClass().toString());
		category.method_577("Error Info gathered from JEI", () -> ingredientHelper.getErrorInfo(ingredient));
	}

	public static String crashReportToString(Throwable t, String title, Collection<class_129> categories) {
		StringBuilder sb = new StringBuilder();
		sb.append(title);
		sb.append(":\n\n");
		for (class_129 category : categories) {
			category.method_574(sb);
			sb.append("\n\n");
		}
		sb.append("-- Stack Trace --\n\n");
		sb.append(ExceptionUtils.getStackTrace(t));
		return sb.toString();
	}
}
