package mezz.jei.gui.ingredients;

import mezz.jei.api.ingredients.IIngredientType;
import mezz.jei.api.ingredients.ITypedIngredient;
import mezz.jei.api.runtime.IIngredientManager;
import mezz.jei.common.config.IngredientSortStage;
import mezz.jei.gui.config.IngredientTypeSortingConfig;
import mezz.jei.gui.config.ModNameSortingConfig;
import net.minecraft.class_1738;
import net.minecraft.class_1792;
import net.minecraft.class_1799;
import net.minecraft.class_2960;
import net.minecraft.class_6862;
import net.minecraft.class_6885.class_6887;
import net.minecraft.class_7923;
import net.minecraft.class_7924;
import java.util.Collection;
import java.util.Comparator;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

public class IngredientSorterComparators {
	private final IIngredientManager ingredientManager;
	private final ModNameSortingConfig modNameSortingConfig;
	private final IngredientTypeSortingConfig ingredientTypeSortingConfig;
	private final Set<String> modNames;

	public IngredientSorterComparators(
		IIngredientManager ingredientManager,
		ModNameSortingConfig modNameSortingConfig,
		IngredientTypeSortingConfig ingredientTypeSortingConfig,
		Set<String> modNames
	) {
		this.ingredientManager = ingredientManager;
		this.modNameSortingConfig = modNameSortingConfig;
		this.ingredientTypeSortingConfig = ingredientTypeSortingConfig;
		this.modNames = modNames;
	}

	public Comparator<IListElementInfo<?>> getComparator(List<IngredientSortStage> ingredientSorterStages) {
		return ingredientSorterStages.stream()
			.map(this::getComparator)
			.reduce(Comparator::thenComparing)
			.orElseGet(this::getDefault);
	}

	public Comparator<IListElementInfo<?>> getComparator(IngredientSortStage ingredientSortStage) {
		return switch (ingredientSortStage) {
			case ALPHABETICAL -> getAlphabeticalComparator();
			case CREATIVE_MENU -> getCreativeMenuComparator();
			case INGREDIENT_TYPE -> getIngredientTypeComparator();
			case MOD_NAME -> getModNameComparator();
			case TAG -> getTagComparator();
			case ARMOR -> getArmorComparator();
			case MAX_DURABILITY -> getMaxDurabilityComparator();
		};
	}

	public Comparator<IListElementInfo<?>> getDefault() {
		return getModNameComparator()
			.thenComparing(getIngredientTypeComparator())
			.thenComparing(getCreativeMenuComparator());
	}

	private static Comparator<IListElementInfo<?>> getCreativeMenuComparator() {
		return Comparator.comparingInt(IListElementInfo::getCreatedIndex);
	}

	private static Comparator<IListElementInfo<?>> getAlphabeticalComparator() {
		return Comparator.comparing(i -> i.getNames().get(0));
	}

	private Comparator<IListElementInfo<?>> getModNameComparator() {
		return this.modNameSortingConfig.getComparatorFromMappedValues(modNames);
	}

	private Comparator<IListElementInfo<?>> getIngredientTypeComparator() {
		Collection<IIngredientType<?>> ingredientTypes = this.ingredientManager.getRegisteredIngredientTypes();
		Set<String> ingredientTypeStrings = ingredientTypes.stream()
			.map(IngredientTypeSortingConfig::getIngredientTypeString)
			.collect(Collectors.toSet());
		return this.ingredientTypeSortingConfig.getComparatorFromMappedValues(ingredientTypeStrings);
	}

	private static Comparator<IListElementInfo<?>> getMaxDurabilityComparator() {
		Comparator<IListElementInfo<?>> maxDamage =
			Comparator.comparing(o -> getItemStack(o).method_7936());
		return maxDamage.reversed();
	}

	private Comparator<IListElementInfo<?>> getTagComparator() {
		Comparator<IListElementInfo<?>> isTagged =
			Comparator.comparing(this::hasTag);
		Comparator<IListElementInfo<?>> tag =
			Comparator.comparing(this::getTagForSorting);
		return isTagged.reversed().thenComparing(tag);
	}

	private static Comparator<IListElementInfo<?>> getArmorComparator() {
		Comparator<IListElementInfo<?>> isArmorComp =
			Comparator.comparing(o -> isArmor(getItemStack(o)));
		Comparator<IListElementInfo<?>> armorSlot =
			Comparator.comparing(o -> getArmorSlotIndex(getItemStack(o)));
		Comparator<IListElementInfo<?>> armorDamage =
			Comparator.comparing(o -> getArmorDamageReduce(getItemStack(o)));
		Comparator<IListElementInfo<?>> armorToughness =
			Comparator.comparing(o -> getArmorToughness(getItemStack(o)));
		Comparator<IListElementInfo<?>> maxDamage =
			Comparator.comparing(o -> getArmorDurability(getItemStack(o)));
		return isArmorComp.reversed()
			.thenComparing(armorSlot.reversed())
			.thenComparing(armorDamage.reversed())
			.thenComparing(armorToughness.reversed())
			.thenComparing(maxDamage.reversed());
	}

	private static boolean isArmor(class_1799 itemStack) {
		class_1792 item = itemStack.method_7909();
		return item instanceof class_1738;
	}

	private static int getArmorSlotIndex(class_1799 itemStack) {
		class_1792 item = itemStack.method_7909();
		if (item instanceof class_1738 armorItem) {
			return armorItem.method_7685().method_5926();
		}
		return 0;
	}

	private static int getArmorDamageReduce(class_1799 itemStack) {
		class_1792 item = itemStack.method_7909();
		if (item instanceof class_1738 armorItem) {
			return armorItem.method_7687();
		}
		return 0;
	}

	private static float getArmorToughness(class_1799 itemStack) {
		class_1792 item = itemStack.method_7909();
		if (item instanceof class_1738 armorItem) {
			return armorItem.method_26353();
		}
		return 0;
	}

	private static int getArmorDurability(class_1799 itemStack) {
		if (isArmor(itemStack)) {
			return itemStack.method_7936();
		}
		return 0;
	}

	private String getTagForSorting(IListElementInfo<?> elementInfo) {
		// Choose the most popular tag it has.
		return elementInfo.getTagIds(ingredientManager)
			.max(Comparator.comparing(IngredientSorterComparators::tagCount))
			.map(class_2960::method_12832)
			.orElse("");
	}

	private static int tagCount(class_2960 tagId) {
		//TODO: make a tag blacklist.
		if (tagId.toString().equals("itemfilters:check_nbt")) {
			return 0;
		}
		class_6862<class_1792> tagKey = class_6862.method_40092(class_7924.field_41197, tagId);
		return class_7923.field_41178.method_40266(tagKey)
			.map(class_6887::method_40247)
			.orElse(0);
	}

	private boolean hasTag(IListElementInfo<?> elementInfo) {
		return !getTagForSorting(elementInfo).isEmpty();
	}

	public static <V> class_1799 getItemStack(IListElementInfo<V> ingredientInfo) {
		ITypedIngredient<V> ingredient = ingredientInfo.getTypedIngredient();
		if (ingredient.getIngredient() instanceof class_1799 itemStack) {
			return itemStack;
		}
		return class_1799.field_8037;
	}
}
