package mezz.jei.library.plugins.vanilla.ingredients.fluid;

import com.google.common.base.MoreObjects;
import mezz.jei.api.helpers.IColorHelper;
import mezz.jei.api.ingredients.IIngredientHelper;
import mezz.jei.api.ingredients.IIngredientType;
import mezz.jei.api.ingredients.IIngredientTypeWithSubtypes;
import mezz.jei.api.ingredients.subtypes.ISubtypeManager;
import mezz.jei.api.ingredients.subtypes.UidContext;
import mezz.jei.common.platform.IPlatformFluidHelperInternal;
import mezz.jei.common.platform.IPlatformRegistry;
import mezz.jei.common.platform.Services;
import mezz.jei.common.util.TagUtil;
import net.minecraft.class_1792;
import net.minecraft.class_1799;
import net.minecraft.class_2561;
import net.minecraft.class_2960;
import net.minecraft.class_3611;
import net.minecraft.class_6862;
import net.minecraft.class_6880;
import net.minecraft.class_7923;
import net.minecraft.class_7924;
import org.jetbrains.annotations.Nullable;

import java.util.Collection;
import java.util.List;
import java.util.Optional;
import java.util.stream.Stream;

public class FluidIngredientHelper<T> implements IIngredientHelper<T> {
	private final ISubtypeManager subtypeManager;
	private final IColorHelper colorHelper;
	private final IPlatformFluidHelperInternal<T> platformFluidHelper;
	private final IPlatformRegistry<class_3611> registry;
	private final IIngredientTypeWithSubtypes<class_3611, T> fluidType;

	public FluidIngredientHelper(ISubtypeManager subtypeManager, IColorHelper colorHelper, IPlatformFluidHelperInternal<T> platformFluidHelper) {
		this.subtypeManager = subtypeManager;
		this.colorHelper = colorHelper;
		this.platformFluidHelper = platformFluidHelper;
		this.registry = Services.PLATFORM.getRegistry(class_7924.field_41270);
		this.fluidType = platformFluidHelper.getFluidIngredientType();
	}

	@Override
	public IIngredientType<T> getIngredientType() {
		return platformFluidHelper.getFluidIngredientType();
	}

	@Override
	public String getDisplayName(T ingredient) {
		class_2561 displayName = platformFluidHelper.getDisplayName(ingredient);
		return displayName.getString();
	}

	@Override
	public String getUniqueId(T ingredient, UidContext context) {
		class_3611 fluid = fluidType.getBase(ingredient);
		class_2960 registryName = getRegistryName(ingredient, fluid);

		StringBuilder result = new StringBuilder()
			.append("fluid:")
			.append(registryName);

		String subtypeInfo = subtypeManager.getSubtypeInfo(fluidType, ingredient, context);
		if (!subtypeInfo.isEmpty()) {
			result.append(":");
			result.append(subtypeInfo);
		}

		return result.toString();
	}

	@Override
	public String getWildcardId(T ingredient) {
		class_3611 fluid = fluidType.getBase(ingredient);
		class_2960 registryName = getRegistryName(ingredient, fluid);
		return "fluid:" + registryName;
	}

	@Override
	public Iterable<Integer> getColors(T ingredient) {
		return platformFluidHelper.getStillFluidSprite(ingredient)
			.map(fluidStillSprite -> {
				int renderColor = platformFluidHelper.getColorTint(ingredient);
				return colorHelper.getColors(fluidStillSprite, renderColor, 1);
			})
			.orElseGet(List::of);
	}

	@Override
	public class_2960 getResourceLocation(T ingredient) {
		class_3611 fluid = fluidType.getBase(ingredient);
		return getRegistryName(ingredient, fluid);
	}

	private class_2960 getRegistryName(T ingredient, class_3611 fluid) {
		return registry.getRegistryName(fluid)
			.orElseThrow(() -> {
				String ingredientInfo = getErrorInfo(ingredient);
				return new IllegalStateException("null registry name for: " + ingredientInfo);
			});
	}

	@Override
	public class_1799 getCheatItemStack(T ingredient) {
		class_3611 fluid = fluidType.getBase(ingredient);
		class_1792 filledBucket = fluid.method_15774();
		return new class_1799(filledBucket);
	}

	@Override
	public T copyIngredient(T ingredient) {
		return platformFluidHelper.copy(ingredient);
	}

	@Override
	public T normalizeIngredient(T ingredient) {
		return platformFluidHelper.normalize(ingredient);
	}

	@Override
	public Stream<class_2960> getTagStream(T ingredient) {
		class_3611 fluid = fluidType.getBase(ingredient);

		return class_7923.field_41173.method_29113(fluid)
			.flatMap(class_7923.field_41173::method_40264)
			.map(class_6880::method_40228)
			.orElse(Stream.of())
			.map(class_6862::comp_327);
	}

	@SuppressWarnings("ConstantConditions")
	@Override
	public String getErrorInfo(@Nullable T ingredient) {
		if (ingredient == null) {
			return "null";
		}
		MoreObjects.ToStringHelper toStringHelper = MoreObjects.toStringHelper(ingredient.getClass());
		class_3611 fluid = fluidType.getBase(ingredient);
		if (fluid != null) {
			class_2561 displayName = platformFluidHelper.getDisplayName(ingredient);
			toStringHelper.add("Fluid", displayName.getString());
		} else {
			toStringHelper.add("Fluid", "null");
		}

		toStringHelper.add("Amount", platformFluidHelper.getAmount(ingredient));

		platformFluidHelper.getTag(ingredient)
			.ifPresent(tag -> toStringHelper.add("Tag", tag));

		return toStringHelper.toString();
	}

	@Override
	public Optional<class_2960> getTagEquivalent(Collection<T> ingredients) {
		return TagUtil.getTagEquivalent(ingredients, fluidType::getBase, class_7923.field_41173::method_40272);
	}
}
