package mezz.jei.common.transfer;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import javax.annotation.Nonnull;
import net.minecraft.class_1657;
import net.minecraft.class_1703;
import net.minecraft.class_1735;
import net.minecraft.class_1799;
import org.jetbrains.annotations.Nullable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

public final class BasicRecipeTransferHandlerServer {
	private static final Logger LOGGER = LogManager.getLogger();

	private BasicRecipeTransferHandlerServer() {
	}

	/**
	 * Called server-side to actually put the items in place.
	 */
	public static void setItems(
		class_1657 player,
		List<TransferOperation> transferOperations,
		List<class_1735> craftingSlots,
		List<class_1735> inventorySlots,
		boolean maxTransfer,
		boolean requireCompleteSets
	) {
		if (!RecipeTransferUtil.validateSlots(player, transferOperations, craftingSlots, inventorySlots)) {
			return;
		}

		Map<class_1735, ItemStackWithSlotHint> recipeSlotToRequiredItemStack = calculateRequiredStacks(transferOperations, player);
		if (recipeSlotToRequiredItemStack == null) {
			return;
		}

		// Transfer as many items as possible only if it has been explicitly requested by the implementation
		// and a max-transfer operation has been requested by the player.
		boolean transferAsCompleteSets = requireCompleteSets || !maxTransfer;

		Map<class_1735, class_1799> recipeSlotToTakenStacks = takeItemsFromInventory(
			player,
			recipeSlotToRequiredItemStack,
			craftingSlots,
			inventorySlots,
			transferAsCompleteSets,
			maxTransfer
		);

		if (recipeSlotToTakenStacks.isEmpty()) {
			LOGGER.error("Tried to transfer recipe but was unable to remove any items from the inventory.");
			return;
		}

		// clear the crafting grid
		List<class_1799> clearedCraftingItems = clearCraftingGrid(craftingSlots, player);

		// put items into the crafting grid
		List<class_1799> remainderItems = putItemsIntoCraftingGrid(recipeSlotToTakenStacks, requireCompleteSets);

		// put leftover items back into the inventory
		stowItems(player, inventorySlots, clearedCraftingItems);
		stowItems(player, inventorySlots, remainderItems);

		class_1703 container = player.field_7512;
		container.method_7623();
	}

	private static int getSlotStackLimit(
		Map<class_1735, class_1799> recipeSlotToTakenStacks,
		boolean requireCompleteSets
	) {
		if (!requireCompleteSets) {
			return Integer.MAX_VALUE;
		}

		return recipeSlotToTakenStacks.entrySet().stream()
			.mapToInt(e -> {
				class_1735 craftingSlot = e.getKey();
				class_1799 transferItem = e.getValue();
				if (craftingSlot.method_7680(transferItem)) {
					return craftingSlot.method_7676(transferItem);
				}
				return Integer.MAX_VALUE;
			})
			.min()
			.orElse(Integer.MAX_VALUE);
	}

	private static List<class_1799> clearCraftingGrid(List<class_1735> craftingSlots, class_1657 player) {
		List<class_1799> clearedCraftingItems = new ArrayList<>();
		for (class_1735 craftingSlot : craftingSlots) {
			if (!craftingSlot.method_7674(player)) {
				continue;
			}
			if (craftingSlot.method_7681()) {
				class_1799 craftingItem = craftingSlot.method_7671(Integer.MAX_VALUE);
				clearedCraftingItems.add(craftingItem);
			}
		}
		return clearedCraftingItems;
	}

	private static List<class_1799> putItemsIntoCraftingGrid(
		Map<class_1735, class_1799> recipeSlotToTakenStacks,
		boolean requireCompleteSets
	) {
		final int slotStackLimit = getSlotStackLimit(recipeSlotToTakenStacks, requireCompleteSets);
		List<class_1799> remainderItems = new ArrayList<>();

		recipeSlotToTakenStacks.forEach((slot, stack) -> {
			if (slot.method_7677().method_7960() && slot.method_7680(stack)) {
				class_1799 remainder = slot.method_32755(stack, slotStackLimit);
				if (!remainder.method_7960()) {
					remainderItems.add(remainder);
				}
			} else {
				remainderItems.add(stack);
			}
		});

		return remainderItems;
	}

	@Nullable
	private static Map<class_1735, ItemStackWithSlotHint> calculateRequiredStacks(List<TransferOperation> transferOperations, class_1657 player) {
		Map<class_1735, ItemStackWithSlotHint> recipeSlotToRequired = new HashMap<>(transferOperations.size());
		for (TransferOperation transferOperation : transferOperations) {
			class_1735 recipeSlot = transferOperation.craftingSlot();
			class_1735 inventorySlot = transferOperation.inventorySlot();
			if (!inventorySlot.method_7674(player)) {
				LOGGER.error(
					"Tried to transfer recipe but was given an" +
					" inventory slot that the player can't pickup from: {}" ,
					inventorySlot.field_7874
				);
				return null;
			}
			final class_1799 slotStack = inventorySlot.method_7677();
			if (slotStack.method_7960()) {
				LOGGER.error(
					"Tried to transfer recipe but was given an" +
					" empty inventory slot as an ingredient source: {}",
					inventorySlot.field_7874
				);
				return null;
			}
			class_1799 stack = slotStack.method_7972();
			stack.method_7939(1);
			recipeSlotToRequired.put(recipeSlot, new ItemStackWithSlotHint(inventorySlot, stack));
		}
		return recipeSlotToRequired;
	}

	@Nonnull
	private static Map<class_1735, class_1799> takeItemsFromInventory(
		class_1657 player,
		Map<class_1735, ItemStackWithSlotHint> recipeSlotToRequiredItemStack,
		List<class_1735> craftingSlots,
		List<class_1735> inventorySlots,
		boolean transferAsCompleteSets,
		boolean maxTransfer
	) {
		if (!maxTransfer) {
			return removeOneSetOfItemsFromInventory(
				player,
				recipeSlotToRequiredItemStack,
				craftingSlots,
				inventorySlots,
				transferAsCompleteSets
			);
		}

		final Map<class_1735, class_1799> recipeSlotToResult = new HashMap<>(recipeSlotToRequiredItemStack.size());
		while (true) {
			final Map<class_1735, class_1799> foundItemsInSet = removeOneSetOfItemsFromInventory(
				player,
				recipeSlotToRequiredItemStack,
				craftingSlots,
				inventorySlots,
				transferAsCompleteSets
			);

			if (foundItemsInSet.isEmpty()) {
				break;
			}

			// Merge the contents of the temporary map with the result map.
			Set<class_1735> fullSlots = merge(recipeSlotToResult, foundItemsInSet);

			// to avoid overfilling slots, remove any requirements that have been met
			for (class_1735 fullSlot : fullSlots) {
				recipeSlotToRequiredItemStack.remove(fullSlot);
			}
		}

		return recipeSlotToResult;
	}

	private static Map<class_1735, class_1799> removeOneSetOfItemsFromInventory(
		class_1657 player,
		Map<class_1735, ItemStackWithSlotHint> recipeSlotToRequiredItemStack,
		List<class_1735> craftingSlots,
		List<class_1735> inventorySlots,
		boolean transferAsCompleteSets
	) {
		Map<class_1735, class_1799> originalSlotContents = null;
		if (transferAsCompleteSets) {
			// We only need to create a new map for each set iteration if we're transferring as complete sets.
			originalSlotContents = new HashMap<>();
		}

		// This map holds items found for each set iteration. Its contents are added to the result map
		// after each complete set iteration. If we are transferring as complete sets, this allows
		// us to simply ignore the map's contents when a complete set isn't found.
		final Map<class_1735, class_1799> foundItemsInSet = new HashMap<>(recipeSlotToRequiredItemStack.size());

		for (Map.Entry<class_1735, ItemStackWithSlotHint> entry : recipeSlotToRequiredItemStack.entrySet()) { // for each item in set
			final class_1735 recipeSlot = entry.getKey();
			final class_1799 requiredStack = entry.getValue().stack;
			final class_1735 hint = entry.getValue().hint;

			// Locate a slot that has what we need.
			final class_1735 slot = getSlotWithStack(player, requiredStack, craftingSlots, inventorySlots, hint)
				.orElse(null);
			if (slot != null) {
				// the item was found

				// Keep a copy of the slot's original contents in case we need to roll back.
				if (originalSlotContents != null && !originalSlotContents.containsKey(slot)) {
					originalSlotContents.put(slot, slot.method_7677().method_7972());
				}

				// Reduce the size of the found slot.
				class_1799 removedItemStack = slot.method_7671(1);
				foundItemsInSet.put(recipeSlot, removedItemStack);
			} else {
				// We can't find any more slots to fulfill the requirements.

				if (transferAsCompleteSets) {
					// Since the full set requirement wasn't satisfied, we need to roll back any
					// slot changes we've made during this set iteration.
					for (Map.Entry<class_1735, class_1799> slotEntry : originalSlotContents.entrySet()) {
						class_1799 stack = slotEntry.getValue();
						slotEntry.getKey().method_7673(stack);
					}
					return Map.of();
				}
			}
		}
		return foundItemsInSet;
	}

	private static Set<class_1735> merge(Map<class_1735, class_1799> result, Map<class_1735, class_1799> addition) {
		Set<class_1735> fullSlots = new HashSet<>();

		addition.forEach((slot, itemStack) -> {
			assert itemStack.method_7947() == 1;

			class_1799 resultItemStack = result.get(slot);
			if (resultItemStack == null) {
				resultItemStack = itemStack;
				result.put(slot, resultItemStack);
			} else {
				assert class_1799.method_31577(resultItemStack, itemStack);
				resultItemStack.method_7933(itemStack.method_7947());
			}
			if (resultItemStack.method_7947() == slot.method_7676(resultItemStack)) {
				fullSlots.add(slot);
			}
		});

		return fullSlots;
	}

	private static Optional<class_1735> getSlotWithStack(class_1657 player, class_1799 stack, List<class_1735> craftingSlots, List<class_1735> inventorySlots, class_1735 hint) {
		return getSlotWithStack(player, craftingSlots, stack)
			.or(() -> getValidatedHintSlot(player, stack, hint))
			.or(() -> getSlotWithStack(player, inventorySlots, stack));
	}

	private static Optional<class_1735> getValidatedHintSlot(class_1657 player, class_1799 stack, class_1735 hint) {
		if (hint.method_7674(player) &&
			!hint.method_7677().method_7960() &&
			class_1799.method_31577(stack, hint.method_7677())
		) {
			return Optional.of(hint);
		}

		return Optional.empty();
	}

	private static void stowItems(class_1657 player, List<class_1735> inventorySlots, List<class_1799> itemStacks) {
		for (class_1799 itemStack : itemStacks) {
			class_1799 remainder = stowItem(inventorySlots, itemStack);
			if (!remainder.method_7960()) {
				if (!player.method_31548().method_7394(remainder)) {
					player.method_7328(remainder, false);
				}
			}
		}
	}

	private static class_1799 stowItem(Collection<class_1735> slots, class_1799 stack) {
		if (stack.method_7960()) {
			return class_1799.field_8037;
		}

		final class_1799 remainder = stack.method_7972();

		// Add to existing stacks first
		for (class_1735 slot : slots) {
			final class_1799 inventoryStack = slot.method_7677();
			if (!inventoryStack.method_7960() && inventoryStack.method_7946()) {
				slot.method_32756(remainder);
				if (remainder.method_7960()) {
					return class_1799.field_8037;
				}
			}
		}

		// Try adding to empty slots
		for (class_1735 slot : slots) {
			if (slot.method_7677().method_7960()) {
				slot.method_32756(remainder);
				if (remainder.method_7960()) {
					return class_1799.field_8037;
				}
			}
		}

		return remainder;
	}

	/**
	 * Get the slot which contains a specific itemStack.
	 *
	 * @param slots     the slots in the container to search
	 * @param itemStack the itemStack to find
	 * @return the slot that contains the itemStack. returns null if no slot contains the itemStack.
	 */
	private static Optional<class_1735> getSlotWithStack(class_1657 player, Collection<class_1735> slots, class_1799 itemStack) {
		return slots.stream()
			.filter(slot -> {
				class_1799 slotStack = slot.method_7677();
				return class_1799.method_31577(itemStack, slotStack) &&
					slot.method_7674(player);
			})
			.findFirst();
	}

	private record ItemStackWithSlotHint(class_1735 hint, class_1799 stack) {}
}
