/*
 * JourneyMap API (http://journeymap.info)
 * http://github.com/TeamJM/journeymap-api
 *
 * Copyright (c) 2011-2016 Techbrew.  All Rights Reserved.
 * The following limited rights are granted to you:
 *
 * You MAY:
 *  + Write your own code that uses the API source code in journeymap.* packages as a dependency.
 *  + Write and distribute your own code that uses, modifies, or extends the example source code in example.* packages
 *  + Fork and modify any source code for the purpose of submitting Pull Requests to the TeamJM/journeymap-api repository.
 *    Submitting new or modified code to the repository means that you are granting Techbrew all rights to the submitted code.
 *
 * You MAY NOT:
 *  - Distribute source code or classes (whether modified or not) from journeymap.* packages.
 *  - Submit any code to the TeamJM/journeymap-api repository with a different license than this one.
 *  - Use code or artifacts from the repository in any way not explicitly granted by this license.
 *
 */

package journeymap.api.v2.client.util;

import journeymap.api.v2.client.model.MapPolygon;
import journeymap.api.v2.client.model.MapPolygonWithHoles;
import net.minecraft.class_1923;
import net.minecraft.class_2338;
import net.minecraft.class_3545;
import javax.annotation.Nonnull;
import java.awt.Polygon;
import java.awt.Rectangle;
import java.awt.geom.Area;
import java.awt.geom.PathIterator;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.stream.Collectors;

/**
 * Utility class related to Polygons.
 */
public class PolygonHelper
{
    /**
     * Creates a polygon for the chunk containing worldCoords, starting with the lower-left (southwest) corner
     * and going counter-clockwise.
     *
     * @param x world X
     * @param y world Y
     * @param z world Z
     * @return a polygon for the surrounding chunk
     */
    public static MapPolygon createChunkPolygonForWorldCoords(int x, int y, int z)
    {
        return createChunkPolygon(x >> 4, y, z >> 4);
    }

    /**
     * Creates a polygon for the chunk coords, starting with the lower-left (southwest) corner
     * and going counter-clockwise.
     *
     * @param chunkX chunk x
     * @param y      block y
     * @param chunkZ chunk z
     * @return polygon
     */
    public static MapPolygon createChunkPolygon(int chunkX, int y, int chunkZ)
    {
        int x = chunkX << 4;
        int z = chunkZ << 4;
        class_2338 sw = new class_2338(x, y, z + 16);
        class_2338 se = new class_2338(x + 16, y, z + 16);
        class_2338 ne = new class_2338(x + 16, y, z);
        class_2338 nw = new class_2338(x, y, z);

        return new MapPolygon(sw, se, ne, nw);
    }

    /**
     * Creates a polygon for the block coords, starting with the lower-left (southwest) corner
     * and going counter-clockwise.  The supplied coordinates don't need to be in order, just opposite.
     *
     * @param corner1 One corner of the desired rectangle
     * @param corner2 The opposite corner
     * @return polygon
     */
    public static MapPolygon createBlockRect(final class_2338 corner1, final class_2338 corner2)
    {
        final int minX = Math.min(corner1.method_10263(), corner2.method_10263());
        final int maxX = Math.max(corner1.method_10263(), corner2.method_10263());
        final int minZ = Math.min(corner1.method_10260(), corner2.method_10260());
        final int maxZ = Math.max(corner1.method_10260(), corner2.method_10260());

        final class_2338 sw = new class_2338(minX, corner1.method_10264(), maxZ);
        final class_2338 se = new class_2338(maxX, corner1.method_10264(), maxZ);
        final class_2338 ne = new class_2338(maxX, corner2.method_10264(), minZ);
        final class_2338 nw = new class_2338(minX, corner2.method_10264(), minZ);

        return new MapPolygon(sw, se, ne, nw);
    }

    /**
     * Given a collection of chunks, creates an {@link Area} that covers them.
     *
     * @param chunks The set of chunks.
     * @return An Area of the corresponding block coordinates.
     */
    @Nonnull
    public static Area createChunksArea(@Nonnull final Collection<class_1923> chunks)
    {
        final Area area = new Area();
        for (final class_1923 chunkPos : chunks)
        {
            area.add(new Area(new Rectangle(chunkPos.method_8326(), chunkPos.method_8328(), 16, 16)));
        }
        return area;
    }

    /**
     * Given a collection of chunks, creates one or more {@link MapPolygonWithHoles} that covers them.
     * (Just a convenience wrapper for the {@link Area}-based methods.)
     *
     * @param chunks The set of chunks.
     * @param y      The y-coordinate for the resulting polygons.
     * @return One or more polygons that cover the specified chunks.
     */
    @Nonnull
    public static List<MapPolygonWithHoles> createChunksPolygon(@Nonnull final Collection<class_1923> chunks, final int y)
    {
        return createPolygonFromArea(createChunksArea(chunks), y);
    }

    /**
     * Converts a {@link MapPolygon} into an {@link Area} (keeping XZ coords only).
     *
     * @param polygon The polygon.
     * @return The corresponding area.
     */
    @Nonnull
    public static Area toArea(@Nonnull final MapPolygon polygon)
    {
        final List<class_2338> points = polygon.getPoints();
        final int[] xPoints = new int[points.size()];
        final int[] yPoints = new int[points.size()];

        for (int i = 0; i < points.size(); ++i)
        {
            xPoints[i] = points.get(i).method_10263();
            yPoints[i] = points.get(i).method_10260();
        }

        return new Area(new Polygon(xPoints, yPoints, points.size()));
    }

    /**
     * Creates a set of {@link MapPolygonWithHoles} from the given {@link Area} (XZ block
     * coords) and the given Y coord.
     * <p>
     * Note that this includes some point-simplification that currently only works if the area is
     * only made up of rectangular subregions -- i.e. all lines are perfectly horizontal or vertical.
     * If you do have diagonal lines this is mostly harmless; just might leave more points than are
     * strictly required.
     *
     * @param area The area to cover.
     * @param y    The y-coordinate.
     * @return The polygons.
     */
    @Nonnull
    public static List<MapPolygonWithHoles> createPolygonFromArea(@Nonnull final Area area, final int y)
    {
        final List<MapPolygon> polygons = new ArrayList<>();
        List<class_2338> poly = new ArrayList<>();
        final PathIterator iterator = area.getPathIterator(null);
        final float[] points = new float[6];
        while (!iterator.isDone())
        {
            final int type = iterator.currentSegment(points);
            switch (type)
            {
                case PathIterator.SEG_MOVETO:
                    if (!poly.isEmpty())
                    {
                        poly = simplify(poly);
                        polygons.add(new MapPolygon(poly));
                        poly = new ArrayList<>();
                    }
                    poly.add(new class_2338(Math.round(points[0]), y, Math.round(points[1])));
                    break;
                case PathIterator.SEG_LINETO:
                    poly.add(new class_2338(Math.round(points[0]), y, Math.round(points[1])));
                    break;
            }
            iterator.next();
        }
        if (!poly.isEmpty())
        {
            polygons.add(new MapPolygon(poly));
        }

        return classifyAndGroup(polygons);
    }

    /**
     * Given an arbitrary list of polygons, determine which are hulls and holes and which holes are
     * associated with which hulls.
     * <p>
     * Assumes that hulls use CCW point winding and holes use CW point winding, which seems to be
     * consistent with {@link #createPolygonFromArea}.
     *
     * @param polygons The input list of {@link MapPolygon}s.
     * @return The resulting list of {@link MapPolygonWithHoles}.
     */
    @Nonnull
    public static List<MapPolygonWithHoles> classifyAndGroup(@Nonnull final List<MapPolygon> polygons)
    {
        final List<MapPolygon> hulls = new ArrayList<>();
        final List<MapPolygon> holes = new ArrayList<>();

        for (final MapPolygon polygon : polygons)
        {
            if (isHole(polygon))
            {
                holes.add(polygon);
            }
            else
            {
                hulls.add(polygon);
            }
        }

        final List<class_3545<MapPolygon, Area>> holeAreas = holes.stream()
                .map(hole -> new class_3545<>(hole, toArea(hole)))
                .collect(Collectors.toList());

        final List<MapPolygonWithHoles> result = new ArrayList<>();
        for (final MapPolygon hull : hulls)
        {
            final Area hullArea = toArea(hull);
            final List<MapPolygon> hullHoles = new ArrayList<>();

            for (final Iterator<class_3545<MapPolygon, Area>> iterator = holeAreas.iterator(); iterator.hasNext(); )
            {
                final class_3545<MapPolygon, Area> holeArea = iterator.next();
                final Area intersection = new Area(hullArea);
                intersection.intersect(holeArea.method_15441());
                if (!intersection.isEmpty())
                {
                    hullHoles.add(holeArea.method_15442());
                    iterator.remove();
                }
            }

            result.add(new MapPolygonWithHoles(hull, hullHoles));
        }

        return result;
    }

    /**
     * The input tends to have points for each chunk, even along a straight line.
     * Remove the unneeded intermediate points.  Currently this only works along
     * purely horizontal/vertical lines, not diagonals.
     *
     * @param points The input points
     * @return The filtered points
     */
    @Nonnull
    private static List<class_2338> simplify(@Nonnull final List<class_2338> points)
    {
        final List<class_2338> result = new ArrayList<>();
        class_2338 prev2 = points.get(0);
        class_2338 prev1 = points.get(1);
        result.add(prev2);
        for (int index = 2; index < points.size(); ++index)
        {
            final class_2338 next = points.get(index);
            if (prev2.method_10263() == prev1.method_10263() && prev1.method_10263() == next.method_10263())
            {
                // merge horizontal line by skipping the middle point
                prev1 = next;
            }
            else if (prev2.method_10260() == prev1.method_10260() && prev1.method_10260() == next.method_10260())
            {
                // merge vertical line by skipping the middle point
                prev1 = next;
            }
            else
            {
                // corner; keep the point
                result.add(prev1);
                prev2 = prev1;
                prev1 = next;
            }
        }
        result.add(prev1);
        return result;
    }

    /**
     * Determine if the given polygon is a "hole".  Holes have CW point winding.
     * Assumes that +X is "right" and +Z is "down".
     *
     * @param polygon The polygon.
     * @return True if it's a hole.
     */
    private static boolean isHole(@Nonnull final MapPolygon polygon)
    {
        // from https://stackoverflow.com/a/18472899/43534
        long sum = 0;
        final List<class_2338> points = polygon.getPoints();
        class_2338 a = points.get(points.size() - 1);
        for (final class_2338 b : points)
        {
            sum += (long) (b.method_10263() - a.method_10263()) * (b.method_10260() + a.method_10260());
            a = b;
        }
        return sum < 0;
    }
}
