package journeymap.api.v2.client.util;

import com.google.common.base.MoreObjects;
import com.google.common.base.Objects;
import journeymap.api.v2.client.display.Context;
import net.minecraft.client.Minecraft;
import net.minecraft.core.BlockPos;
import net.minecraft.resources.ResourceKey;
import net.minecraft.world.level.Level;
import net.minecraft.world.phys.AABB;

import javax.annotation.Nullable;
import java.awt.geom.Rectangle2D;

/**
 * Provides the current state of a Map UI in JourneyMap.
 */
public final class UIState
{
    /**
     * Max Zoom In value.
     */
    public final static int ZOOM_IN_MAX = 16384;

    /**
     * Max Zoom Out for fullscreen 2 pixels per region.
     */
    public final static int FULLSCREEN_ZOOM_MIN = 2;

    /**
     * Maz Zoom out for the minimap.
     */
    public final static int MINIMAP_ZOOM_MIN = 256;

    /**
     * The UI to which this state applies.
     */
    public final Context.UI ui;

    /**
     * Whether the UI is active or not. If false, other values reflect the
     * state of the display when it was last used.
     */
    public final boolean active;

    /**
     * The dimension displayed in the UI.  If active==false and the display
     * has never been used, this will default to 0.
     */
    public final ResourceKey<Level> dimension;

    /**
     * The current zoom level of the UI. If active==false and the display
     * has never been used, this will default to 0.
     */
    public final int zoom;

    /**
     * The current map type of the UI. If active==false and the display
     * has never been used, this will default to Context.MapType.Day.
     */
    public final Context.MapType mapType;

    /**
     * The block position at the center of the UI. If active==false and the display
     * has never been used, this will default to the world spawnpoint.
     */
    public final BlockPos mapCenter;

    /**
     * For underground/cave/nether/end maps, the vertical slice (chunk y) displayed.
     */
    public final Integer chunkY;

    /**
     * The area of blocks displayed in the UI. If active==false, this will be null.
     */
    public final AABB blockBounds;

    /**
     * The screen area (pixels) used by the UI.  If active==false, this will be null.
     */
    public final Rectangle2D.Double displayBounds;

    /**
     * The width in pixels of a single block in the UI's map at the current zoom level.
     */
    public final double blockSize;


    /**
     * Constructor.
     *
     * @param ui          The UI which has been updated.
     * @param active      Whether the UI is active
     * @param dimension   The current dimension shown in the UI.
     * @param zoom        The current zoom level of the UI.
     * @param mapType     The current map type of the UI.
     * @param mapCenter   The block position at the center of the UI.
     * @param blockBounds The area of blocks displayed in the UI.
     */
    public UIState(Context.UI ui, boolean active, ResourceKey<Level> dimension, int zoom,
                   @Nullable Context.MapType mapType,
                   @Nullable BlockPos mapCenter,
                   @Nullable Integer chunkY,
                   @Nullable AABB blockBounds,
                   @Nullable Rectangle2D.Double displayBounds)
    {
        this.ui = ui;
        this.active = active;
        this.dimension = dimension;
        this.zoom = zoom;
        this.mapType = mapType;
        this.mapCenter = mapCenter;
        this.chunkY = chunkY;
        this.blockBounds = blockBounds;
        this.displayBounds = displayBounds;
        this.blockSize = zoom / 512.0D;
    }

    /**
     * Convenience factory method to create an inactive UIState.
     *
     * @param ui the ui
     * @return a UIState
     */
    public static UIState newInactive(Context.UI ui, Minecraft minecraft)
    {
        BlockPos center = minecraft.level == null ? new BlockPos(0, 68, 0) : minecraft.level.getSharedSpawnPos(); //getSharedSpawnPos() == getSpawnPoint()
        return new UIState(ui, false, Level.OVERWORLD, 0, Context.MapType.Day, center, null, null, null);
    }

    /**
     * Convenience factory method to create an inactive UIState.
     *
     * @param priorState the prior UIState
     * @return a UIState
     */
    public static UIState newInactive(UIState priorState)
    {
        return new UIState(priorState.ui, false, priorState.dimension, priorState.zoom, priorState.mapType,
                priorState.mapCenter, priorState.chunkY, priorState.blockBounds, priorState.displayBounds);
    }

    @Override
    public boolean equals(Object o)
    {
        if (this == o)
        {
            return true;
        }
        if (o == null || getClass() != o.getClass())
        {
            return false;
        }
        UIState mapState = (UIState) o;
        return Objects.equal(active, mapState.active) &&
                Objects.equal(dimension, mapState.dimension) &&
                Objects.equal(zoom, mapState.zoom) &&
                Objects.equal(ui, mapState.ui) &&
                Objects.equal(mapType, mapState.mapType) &&
                Objects.equal(displayBounds, mapState.displayBounds);
    }

    @Override
    public int hashCode()
    {
        return Objects.hashCode(ui, active, dimension, zoom, mapType, displayBounds);
    }

    @Override
    public String toString()
    {
        return MoreObjects.toStringHelper(this)
                .add("ui", ui)
                .add("active", active)
                .add("dimension", dimension)
                .add("mapType", mapType)
                .add("zoom", zoom)
                .add("mapCenter", mapCenter)
                .add("chunkY", chunkY)
                .add("blockBounds", blockBounds)
                .add("displayBounds", displayBounds)
                .add("blockSize", blockSize)
                .toString();
    }


}
