package foundry.veil.impl.client.render.dynamicbuffer;

import com.google.common.collect.Sets;
import com.mojang.blaze3d.systems.RenderSystem;
import foundry.veil.Veil;
import foundry.veil.api.client.render.VeilRenderSystem;
import foundry.veil.ext.ShaderInstanceExtension;
import foundry.veil.impl.ThreadTaskScheduler;
import foundry.veil.impl.client.render.shader.SimpleShaderProcessor;
import org.apache.commons.io.IOUtils;

import java.io.IOException;
import java.io.Reader;
import java.util.Collection;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import net.minecraft.class_293;
import net.minecraft.class_2960;
import net.minecraft.class_310;
import net.minecraft.class_3300;
import net.minecraft.class_4239;
import net.minecraft.class_5913;
import net.minecraft.class_5944;

import static org.lwjgl.opengl.GL20C.GL_FRAGMENT_SHADER;
import static org.lwjgl.opengl.GL20C.GL_VERTEX_SHADER;

public class VanillaShaderCompiler {

    private static final Set<String> LAST_FRAME_SHADERS = ConcurrentHashMap.newKeySet();

    private ThreadTaskScheduler scheduler;

    public VanillaShaderCompiler() {
    }

    private void compileShader(class_5944 shader) {
        ShaderInstanceExtension extension = (ShaderInstanceExtension) shader;
        Collection<class_2960> shaderSources = extension.veil$getShaderSources();
        class_293 vertexFormat = shader.method_35786();
        class_3300 resourceManager = class_310.method_1551().method_1478();

        SimpleShaderProcessor.setup(resourceManager);
        for (class_2960 path : shaderSources) {
            try (Reader reader = resourceManager.openAsReader(path)) {
                String source = IOUtils.toString(reader);
                class_5913 preprocessor = new class_5913() {
                    private final Set<String> importedPaths = Sets.newHashSet();

                    @Override
                    public String method_34233(boolean useFullPath, String directory) {
                        directory = class_4239.method_34676((useFullPath ? path.method_12832() : "shaders/include/") + directory);
                        if (!this.importedPaths.add(directory)) {
                            return null;
                        } else {
                            class_2960 resourcelocation = class_2960.method_60654(directory);

                            try {
                                String s2;
                                try (Reader reader = resourceManager.openAsReader(resourcelocation)) {
                                    s2 = IOUtils.toString(reader);
                                }

                                return s2;
                            } catch (IOException e) {
                                Veil.LOGGER.error("Could not open GLSL import {}: {}", directory, e.getMessage());
                                return "#error " + e.getMessage();
                            }
                        }
                    }
                };
                source = String.join("", preprocessor.method_34229(source));

                boolean vertex = path.method_12832().endsWith(".vsh");
                String processed = SimpleShaderProcessor.modify(shader.method_35787(), path, vertexFormat, vertex ? GL_VERTEX_SHADER : GL_FRAGMENT_SHADER, source);
                RenderSystem.recordRenderCall(() -> extension.veil$recompile(vertex, processed));
            } catch (Throwable t) {
                Veil.LOGGER.error("Couldn't load vanilla shader from {}", path, t);
            }
        }
        SimpleShaderProcessor.free();
    }

    /**
     * Attempts to preload all vanilla minecraft shader files before creating the shaders on the CPU.
     *
     * @param shaders The shaders to reload
     * @return A future for when vanilla shaders have reloaded
     */
    public CompletableFuture<?> reload(Collection<class_5944> shaders) {
        if (this.scheduler != null) {
            // Cancel the previous tasks and move on
            this.scheduler.cancel();
        }

        Map<String, class_5944> shaderMap = new ConcurrentHashMap<>(shaders.size());
        for (class_5944 shader : shaders) {
            shaderMap.put(shader.method_35787(), shader);
        }

        ThreadTaskScheduler scheduler = new ThreadTaskScheduler("VeilShaderCompile", Math.max(1, Runtime.getRuntime().availableProcessors() / 4), () -> {
            for (String lastFrameShader : LAST_FRAME_SHADERS) {
                class_5944 shader = shaderMap.remove(lastFrameShader);
                if (shader != null) {
                    return () -> this.compileShader(shader);
                }
            }

            Iterator<class_5944> iterator = shaderMap.values().iterator();
            if (iterator.hasNext()) {
                class_5944 shader = iterator.next();
                iterator.remove();
                return () -> this.compileShader(shader);
            }
            return null;
        });
        this.scheduler = scheduler;
        CompletableFuture<?> future = scheduler.getCompletedFuture();
        future.thenRunAsync(() -> {
            if (!scheduler.isCancelled()) {
                Veil.LOGGER.info("Compiled {} vanilla shaders", shaders.size());
            }
        }, class_310.method_1551());
        return future;
    }

    public boolean isCompilingShaders() {
        return this.scheduler != null && !this.scheduler.getCompletedFuture().isDone();
    }

    public static void markRendered(String shaderInstace) {
        if (VeilRenderSystem.renderer().getVanillaShaderCompiler().isCompilingShaders()) {
            LAST_FRAME_SHADERS.add(shaderInstace);
        }
    }

    public static void clear() {
        LAST_FRAME_SHADERS.clear();
    }
}
