package foundry.veil.api.client.render;

import com.mojang.blaze3d.systems.RenderSystem;
import foundry.veil.api.client.editor.EditorManager;
import foundry.veil.api.client.render.deferred.VeilDeferredRenderer;
import foundry.veil.api.client.render.dynamicbuffer.DynamicBufferType;
import foundry.veil.api.client.render.framebuffer.FramebufferManager;
import foundry.veil.api.client.render.post.PostPipeline;
import foundry.veil.api.client.render.post.PostProcessingManager;
import foundry.veil.api.client.render.rendertype.DynamicRenderTypeManager;
import foundry.veil.api.client.render.shader.ShaderManager;
import foundry.veil.api.client.render.shader.ShaderModificationManager;
import foundry.veil.api.client.render.shader.definition.ShaderPreDefinitions;
import foundry.veil.api.quasar.particle.ParticleSystemManager;
import foundry.veil.ext.LevelRendererExtension;
import foundry.veil.impl.client.imgui.VeilImGuiImpl;
import foundry.veil.impl.client.render.dynamicbuffer.DynamicBufferManger;
import foundry.veil.mixin.accessor.ReloadableResourceManagerAccessor;
import org.jetbrains.annotations.ApiStatus;
import org.lwjgl.system.NativeResource;

import java.util.List;
import net.minecraft.class_1041;
import net.minecraft.class_310;
import net.minecraft.class_3302;
import net.minecraft.class_3304;

/**
 * Manages the render pipeline for Veil.
 *
 * @author Ocelot
 */
public class VeilRenderer implements NativeResource {

    private final DynamicBufferManger dynamicBufferManger;
    private final ShaderModificationManager shaderModificationManager;
    private final ShaderPreDefinitions shaderPreDefinitions;
    private final ShaderManager shaderManager;
    private final FramebufferManager framebufferManager;
    private final PostProcessingManager postProcessingManager;
    private final VeilDeferredRenderer deferredRenderer;
    private final DynamicRenderTypeManager dynamicRenderTypeManager;
    private final ParticleSystemManager quasarParticleManager;
    private final EditorManager editorManager;
    private final CameraMatrices cameraMatrices;
    private final GuiInfo guiInfo;

    @ApiStatus.Internal
    public VeilRenderer(class_3304 resourceManager, class_1041 window) {
        this.dynamicBufferManger = new DynamicBufferManger(window.method_4489(), window.method_4506());
        this.shaderModificationManager = new ShaderModificationManager();
        this.shaderPreDefinitions = new ShaderPreDefinitions();
        this.shaderManager = new ShaderManager(ShaderManager.PROGRAM_SET, this.shaderPreDefinitions, this.dynamicBufferManger);
        this.framebufferManager = new FramebufferManager();
        this.postProcessingManager = new PostProcessingManager();
        ShaderManager deferredShaderManager = new ShaderManager(ShaderManager.DEFERRED_SET, this.shaderPreDefinitions, this.dynamicBufferManger);
        this.deferredRenderer = new VeilDeferredRenderer(deferredShaderManager, this.shaderPreDefinitions, this.framebufferManager, this.postProcessingManager);
        this.dynamicRenderTypeManager = new DynamicRenderTypeManager();
        this.quasarParticleManager = new ParticleSystemManager();
        this.editorManager = new EditorManager(resourceManager);
        this.cameraMatrices = new CameraMatrices();
        this.guiInfo = new GuiInfo();

        List<class_3302> listeners = ((ReloadableResourceManagerAccessor) resourceManager).getListeners();

        // This must finish loading before the game renderer so modifications can apply on load
        listeners.add(0, this.shaderModificationManager);
        // This must be before vanilla shaders so vanilla shaders can be replaced
        listeners.add(1, this.shaderManager);
        resourceManager.method_14477(this.framebufferManager);
        resourceManager.method_14477(this.postProcessingManager);
//        resourceManager.registerReloadListener(this.deferredRenderer);
        resourceManager.method_14477(this.dynamicRenderTypeManager);
    }

    /**
     * Enables the specified dynamic render buffers.
     *
     * @param buffers The buffers to enable
     * @return Whether any change occurred
     */
    public boolean enableBuffers(DynamicBufferType... buffers) {
        RenderSystem.assertOnRenderThreadOrInit();
        if (buffers.length == 0) {
            return false;
        }

        int active = this.dynamicBufferManger.getActiveBuffers() | DynamicBufferType.encode(buffers);
        return this.dynamicBufferManger.setActiveBuffers(active);
    }

    /**
     * Disables the specified dynamic render buffers.
     *
     * @param buffers The buffers to disable
     * @return Whether any change occurred
     */
    public boolean disableBuffers(DynamicBufferType... buffers) {
        RenderSystem.assertOnRenderThreadOrInit();
        if (buffers.length == 0) {
            return false;
        }

        int active = this.dynamicBufferManger.getActiveBuffers() & ~DynamicBufferType.encode(buffers);
        return this.dynamicBufferManger.setActiveBuffers(active);
    }

    /**
     * @return The manger for all dynamically added framebuffer attachments
     */
    public DynamicBufferManger getDynamicBufferManger() {
        return this.dynamicBufferManger;
    }

    /**
     * @return The manager for all custom shader modifications
     */
    public ShaderModificationManager getShaderModificationManager() {
        return this.shaderModificationManager;
    }

    /**
     * @return The set of shader pre-definitions. Changes are automatically synced the next frame
     */
    public ShaderPreDefinitions getShaderDefinitions() {
        return this.shaderPreDefinitions;
    }

    /**
     * @return The manager for all veil shaders
     */
    public ShaderManager getShaderManager() {
        return this.shaderManager;
    }

    /**
     * @return The manager for all custom veil framebuffers
     */
    public FramebufferManager getFramebufferManager() {
        return this.framebufferManager;
    }

    /**
     * @return The manager for all {@link PostPipeline} instances
     */
    public PostProcessingManager getPostProcessingManager() {
        return this.postProcessingManager;
    }

    /**
     * @return The deferred renderer instance
     */
    public VeilDeferredRenderer getDeferredRenderer() {
        return this.deferredRenderer;
    }

    /**
     * @return The manager for all data-driven render types
     */
    public DynamicRenderTypeManager getDynamicRenderTypeManager() {
        return this.dynamicRenderTypeManager;
    }

    /**
     * @return The manager for all quasar particles
     */
    public ParticleSystemManager getParticleManager() {
        return this.quasarParticleManager;
    }

    /**
     * @return The manager for all editors
     */
    public EditorManager getEditorManager() {
        return this.editorManager;
    }

    /**
     * @return The camera matrices instance
     */
    public CameraMatrices getCameraMatrices() {
        return this.cameraMatrices;
    }

    /**
     * @return The gui info instance
     */
    public GuiInfo getGuiInfo() {
        return this.guiInfo;
    }

    /**
     * @return Whether ImGui can be used
     */
    public static boolean hasImGui() {
        return VeilImGuiImpl.get() instanceof VeilImGuiImpl;
    }

    /**
     * @return The culling frustum for the renderer
     */
    public static CullFrustum getCullingFrustum() {
        return ((LevelRendererExtension) class_310.method_1551().field_1769).veil$getCullFrustum();
    }

    @Override
    public void free() {
        this.dynamicBufferManger.free();
        this.shaderManager.close();
        this.framebufferManager.free();
        this.postProcessingManager.free();
        this.deferredRenderer.free();
        this.quasarParticleManager.clear();
        this.cameraMatrices.free();
        this.guiInfo.free();
    }
}
