package foundry.veil.api.client.editor;

import foundry.veil.Veil;
import foundry.veil.api.client.render.VeilRenderer;
import foundry.veil.impl.client.imgui.VeilImGuiImpl;
import imgui.ImFont;
import imgui.ImFontAtlas;
import imgui.ImFontConfig;
import imgui.ImGui;
import org.jetbrains.annotations.ApiStatus;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.lwjgl.system.NativeResource;

import java.io.IOException;
import java.io.InputStream;
import java.text.DecimalFormat;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import net.minecraft.class_2960;
import net.minecraft.class_3298;
import net.minecraft.class_3300;
import net.minecraft.class_3302;
import net.minecraft.class_3695;
import net.minecraft.class_7654;

@ApiStatus.Internal
public class EditorFontManager implements class_3302 {

    private static final class_7654 FONT_LISTER = new class_7654("font", ".ttf");
    private static final DecimalFormat FONT_FORMAT = new DecimalFormat("0.#");
    private static final float FONT_SIZE = 20.0f;

    private final Map<class_2960, FontPackBuilder> fontBuilders;
    private final Map<class_2960, FontPack> fonts;
    private ImFont defaultFont;

    public EditorFontManager() {
        this.fontBuilders = new HashMap<>();
        this.fonts = new HashMap<>();
    }

    public ImFont getFont(class_2960 name, boolean bold, boolean italic) {
        FontPack font = this.fonts.get(name);
        if (font == null) {
            return this.defaultFont;
        }

        if (italic ^ bold) {
            return italic ? font.italic : font.bold;
        } else {
            return italic ? font.boldItalic : font.regular;
        }
    }

    @Override
    public @NotNull CompletableFuture<Void> method_25931(@NotNull class_4045 preparationBarrier, @NotNull class_3300 resourceManager, @NotNull class_3695 prepareProfiler, @NotNull class_3695 applyProfiler, @NotNull Executor backgroundExecutor, @NotNull Executor gameExecutor) {
        if (!VeilRenderer.hasImGui()) {
            return preparationBarrier.method_18352(null);
        }

        return CompletableFuture.supplyAsync(() -> {
            Map<class_2960, FontData> fontData = new HashMap<>();
            for (Map.Entry<class_2960, class_3298> entry : FONT_LISTER.method_45113(resourceManager).entrySet()) {
                class_2960 id = FONT_LISTER.method_45115(entry.getKey());
                class_3298 resource = entry.getValue();
                try (InputStream stream = resource.method_14482()) {
                    short[] ranges = resource.method_14481().method_43041(ImGuiFontMetadataSectionSerializer.INSTANCE)
                            .map(ImGuiFontMetadataSectionSerializer.FontMetadata::ranges)
                            .orElseGet(() -> new short[]{0x0020, 0x00FF, 0});
                    fontData.put(id, new FontData(stream.readAllBytes(), ranges));
                } catch (IOException e) {
                    Veil.LOGGER.error("Failed to load ImGui font: {}", id, e);
                }
            }
            return fontData;
        }, backgroundExecutor).thenCompose(preparationBarrier::method_18352).thenAcceptAsync(fontData -> {
            this.fontBuilders.clear();
            for (Map.Entry<class_2960, FontData> entry : fontData.entrySet()) {
                class_2960 id = entry.getKey();
                String[] parts = id.method_12832().split("-", 2);
                if (parts.length < 2) {
                    continue;
                }

                class_2960 name = class_2960.method_43902(id.method_12836(), parts[0]);
                if (name == null) {
                    Veil.LOGGER.error("Invalid font name: {}:{}", id.method_12836(), parts[0]);
                    continue;
                }

                String type = parts[1];
                FontPackBuilder builder = this.fontBuilders.computeIfAbsent(name, FontPackBuilder::new);
                switch (type) {
                    case "regular" -> builder.main = entry.getValue();
                    case "italic" -> builder.italic = entry.getValue();
                    case "bold" -> builder.bold = entry.getValue();
                    case "bold_italic" -> builder.boldItalic = entry.getValue();
                    default -> Veil.LOGGER.warn("Unknown font type {} for font: {}", type, name);
                }
            }

            this.fontBuilders.entrySet().removeIf(entry -> {
                if (entry.getValue() == null) {
                    Veil.LOGGER.warn("Skipping invalid font: {}", entry.getKey());
                    return true;
                }
                return false;
            });

            Veil.LOGGER.info("Loaded {} ImGui fonts", this.fontBuilders.size());
            this.rebuildFonts();
        }, gameExecutor);
    }

    public void rebuildFonts() {
        try {
            Veil.beginImGui();
            ImFontAtlas atlas = ImGui.getIO().getFonts();
            atlas.clear();
            this.defaultFont = atlas.addFontDefault();

            this.fonts.clear();
            for (Map.Entry<class_2960, FontPackBuilder> entry : this.fontBuilders.entrySet()) {
                Veil.LOGGER.info("Built {}", entry.getKey());
                this.fonts.put(entry.getKey(), entry.getValue().build(FONT_SIZE));
            }
            ImGui.getIO().setFontDefault(this.getFont(EditorManager.DEFAULT_FONT, false, false));
            VeilImGuiImpl.get().updateFonts();
        } finally {
            Veil.endImGui();
        }
    }

    private record FontPack(ImFont regular, ImFont italic, ImFont bold, ImFont boldItalic) implements NativeResource {

        @Override
        public void free() {
            this.regular.destroy();
            if (this.italic != this.regular) {
                this.italic.destroy();
            }
            if (this.bold != this.regular) {
                this.bold.destroy();
            }
            if (this.boldItalic != this.regular) {
                this.boldItalic.destroy();
            }
        }
    }

    private static class FontPackBuilder {

        private final class_2960 name;
        private FontData main;
        private FontData italic;
        private FontData bold;
        private FontData boldItalic;

        private FontPackBuilder(class_2960 name) {
            this.name = name;
        }

        private ImFont loadOrDefault(@Nullable FontData data, String type, float sizePixels, ImFont defaultFont) {
            if (data == null) {
                return defaultFont;
            } else {
                ImFontAtlas atlas = ImGui.getIO().getFonts();
                ImFontConfig fontConfig = new ImFontConfig();
                try {
                    fontConfig.setName(this.name.method_12832() + " " + type + " " + FONT_FORMAT.format(sizePixels) + " px");
                    fontConfig.setGlyphRanges(data.ranges);
                    return atlas.addFontFromMemoryTTF(data.bytes, sizePixels, fontConfig);
                } finally {
                    fontConfig.destroy();
                }
            }
        }

        public FontPack build(float sizePixels) {
            ImFont main = Objects.requireNonNull(this.loadOrDefault(this.main, "regular", sizePixels, null));
            ImFont italic = this.loadOrDefault(this.italic, "italic", sizePixels, main);
            ImFont bold = this.loadOrDefault(this.bold, "bold", sizePixels, main);
            ImFont boldItalic = this.loadOrDefault(this.boldItalic, "bold_italic", sizePixels, main);
            return new FontPack(main, italic, bold, boldItalic);
        }
    }

    private record FontData(byte[] bytes, short[] ranges) {
    }
}
