package foundry.veil.api.quasar.particle;

import foundry.veil.Veil;
import foundry.veil.api.TickTaskScheduler;
import foundry.veil.api.client.render.CullFrustum;
import foundry.veil.api.client.render.MatrixStack;
import foundry.veil.api.quasar.data.ParticleEmitterData;
import foundry.veil.api.quasar.data.QuasarParticles;
import foundry.veil.impl.TickTaskSchedulerImpl;
import org.jetbrains.annotations.ApiStatus;
import org.jetbrains.annotations.Nullable;
import org.joml.Vector3d;

import java.util.*;
import java.util.concurrent.atomic.AtomicInteger;
import net.minecraft.class_1297;
import net.minecraft.class_2960;
import net.minecraft.class_310;
import net.minecraft.class_3532;
import net.minecraft.class_4184;
import net.minecraft.class_4597;
import net.minecraft.class_638;

public class ParticleSystemManager {

    private static final int MAX_PARTICLES = 10000;
    private static final double PERSISTENT_DISTANCE_SQ = 32.0 * 32.0;
    private static final double REMOVAL_DISTANCE_SQ = 128.0 * 128.0;

    private final List<ParticleEmitter> particleEmitters;
    private final Set<class_2960> invalidEmitters;
    private final AtomicInteger particleCount;

    private class_638 level;
    private TickTaskSchedulerImpl scheduler;

    public ParticleSystemManager() {
        this.particleEmitters = new ArrayList<>();
        this.invalidEmitters = new HashSet<>();

        this.particleCount = new AtomicInteger();
        this.level = null;
        this.scheduler = null;
    }

    @ApiStatus.Internal
    public void setLevel(@Nullable class_638 level) {
        this.clear();
        if (this.scheduler != null) {
            this.scheduler.shutdown();
        }

        this.level = level;
        this.scheduler = new TickTaskSchedulerImpl();
    }

    public @Nullable ParticleEmitter createEmitter(class_2960 name) {
        if (this.level == null) {
            return null;
        }
        ParticleEmitterData data = QuasarParticles.registryAccess().method_33310(QuasarParticles.EMITTER).map(registry -> registry.method_10223(name)).orElse(null);
        if (data == null) {
            if (this.invalidEmitters.add(name)) {
                Veil.LOGGER.error("Unknown Quasar Particle Emitter: {}", name);
            }
            return null;
        }
        return new ParticleEmitter(this, this.level, data);
    }

    public void addParticleSystem(ParticleEmitter particleEmitter) {
        this.scheduler.execute(() -> this.particleEmitters.add(particleEmitter));
    }

    public void clear() {
        for (ParticleEmitter particleEmitter : this.particleEmitters) {
            particleEmitter.onRemoved();
        }
        this.particleEmitters.clear();
    }

    @ApiStatus.Internal
    public void tick() {
        if (this.level == null) {
            return;
        }

        this.scheduler.run();
        this.particleCount.set(0);
        Iterator<ParticleEmitter> iterator = this.particleEmitters.iterator();
        while (iterator.hasNext()) {
            ParticleEmitter emitter = iterator.next();
            emitter.tick();
            if (emitter.isRemoved()) {
                emitter.onRemoved();
                iterator.remove();
                continue;
            }

            this.particleCount.addAndGet(emitter.getParticleCount());
        }
    }

    @ApiStatus.Internal
    public void render(MatrixStack matrixStack, class_4597 bufferSource, class_4184 camera, CullFrustum frustum, float partialTicks) {
        // TODO store emitters per-chunk and fetch them from the renderer

        this.particleEmitters.sort(Comparator.comparingDouble(a -> -a.getPosition().distanceSquared(camera.method_19326().field_1352, camera.method_19326().field_1351, camera.method_19326().field_1350)));
        for (ParticleEmitter emitter : this.particleEmitters) {
            emitter.render(matrixStack, bufferSource, camera, partialTicks);
        }
    }

    /**
     * Attempts to remove particles from the most dense and farthest particle emitters to make room for closer emitters.
     *
     * @param particles The number of particles being spawned
     */
    public void reserve(int particles) {
        int freeSpace = MAX_PARTICLES - this.particleCount.getAndAdd(-particles); // This isn't correct, but it doesn't really matter
        if (particles <= freeSpace) {
            return;
        }

        particles -= freeSpace;
        class_1297 cameraEntity = class_310.method_1551().field_1719;
        for (ParticleEmitter emitter : this.particleEmitters) {
            Vector3d pos = emitter.getPosition();
            double scaleFactor = Math.min(cameraEntity != null ? (cameraEntity.method_5649(pos.x, pos.y, pos.z) - PERSISTENT_DISTANCE_SQ) / REMOVAL_DISTANCE_SQ : 1.0, 1.0);
            if (scaleFactor > 0) {
                particles -= emitter.trim(Math.min(particles, class_3532.method_15384(emitter.getParticleCount() * scaleFactor)));
                if (particles <= 0) {
                    break;
                }
            }
        }
    }

    public class_638 getLevel() {
        return this.level;
    }

    public TickTaskScheduler getScheduler() {
        return this.scheduler;
    }

    public int getEmitterCount() {
        return this.particleEmitters.size();
    }

    public int getParticleCount() {
        return this.particleCount.get();
    }
}
