package foundry.veil.impl.client.editor;

import com.mojang.blaze3d.platform.GlStateManager;
import foundry.veil.Veil;
import foundry.veil.api.client.editor.SingleWindowEditor;
import foundry.veil.api.client.imgui.CodeEditor;
import foundry.veil.api.client.imgui.VeilImGuiUtil;
import foundry.veil.api.client.imgui.VeilLanguageDefinitions;
import foundry.veil.api.client.render.VeilRenderSystem;
import foundry.veil.api.client.render.shader.definition.ShaderPreDefinitions;
import foundry.veil.api.client.render.shader.program.ShaderProgram;
import foundry.veil.impl.client.imgui.VeilImGuiImpl;
import foundry.veil.impl.compat.IrisShaderMap;
import foundry.veil.impl.compat.SodiumShaderMap;
import foundry.veil.mixin.accessor.GameRendererAccessor;
import foundry.veil.mixin.accessor.LevelRendererAccessor;
import foundry.veil.mixin.accessor.PostChainAccessor;
import imgui.ImGui;
import imgui.flag.ImGuiCol;
import imgui.flag.ImGuiInputTextFlags;
import imgui.flag.ImGuiStyleVar;
import imgui.type.ImBoolean;
import imgui.type.ImString;
import it.unimi.dsi.fastutil.ints.Int2IntArrayMap;
import it.unimi.dsi.fastutil.ints.IntOpenHashSet;
import it.unimi.dsi.fastutil.ints.IntSet;
import it.unimi.dsi.fastutil.objects.Object2IntMap;
import it.unimi.dsi.fastutil.objects.Object2IntRBTreeMap;
import org.jetbrains.annotations.ApiStatus;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.*;
import java.util.function.BooleanSupplier;
import java.util.function.ObjIntConsumer;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.regex.PatternSyntaxException;
import net.minecraft.class_2561;
import net.minecraft.class_279;
import net.minecraft.class_280;
import net.minecraft.class_283;
import net.minecraft.class_2960;
import net.minecraft.class_310;
import net.minecraft.class_3300;
import net.minecraft.class_4013;
import net.minecraft.class_5944;

import static org.lwjgl.opengl.GL20C.*;
import static org.lwjgl.opengl.GL32C.GL_GEOMETRY_SHADER;
import static org.lwjgl.opengl.GL40C.GL_TESS_CONTROL_SHADER;
import static org.lwjgl.opengl.GL40C.GL_TESS_EVALUATION_SHADER;
import static org.lwjgl.opengl.GL43C.GL_COMPUTE_SHADER;

@ApiStatus.Internal
public class ShaderEditor extends SingleWindowEditor implements class_4013 {

    public static final class_2561 TITLE = class_2561.method_43471("editor.veil.shader.title");

    private static final class_2561 REFRESH = class_2561.method_43471("editor.veil.shader.button.refresh");
    private static final class_2561 UPLOAD = class_2561.method_43471("editor.veil.shader.button.upload");
    private static final class_2561 SEARCH = class_2561.method_43471("editor.veil.shader.search");
    private static final class_2561 SHADER_PROGRAMS = class_2561.method_43471("editor.veil.shader.shader_programs");
    private static final class_2561 SHADER_DEFINITIONS = class_2561.method_43471("editor.veil.shader.definitions");
    private static final class_2561 SHADER_DEFINITIONS_HINT = class_2561.method_43471("editor.veil.shader.definitions.hint");
    private static final class_2561 OPEN_SOURCE = class_2561.method_43471("editor.veil.shader.open_source");

    private static final Pattern ERROR_PARSER = Pattern.compile("ERROR: (\\d+):(\\d+): (.+)");
    private static final Pattern LINE_DIRECTIVE_PARSER = Pattern.compile("#line\\s+(\\d+)\\s*(\\d+)?");

    private final CodeEditor codeEditor;
    private final Object2IntMap<class_2960> shaders;

    private final ImString programFilterText;
    private Pattern programFilter;
    private SelectedProgram selectedProgram;
    private int selectedTab;

    private final ImString addDefinitionText;
    private final Set<String> removedDefinitions;

    private final ImBoolean editSourceOpen;
    private int editProgramId;
    private int editShaderId;

    public ShaderEditor() {
        this.shaders = new Object2IntRBTreeMap<>((a, b) -> {
            int compare = a.method_12836().compareTo(b.method_12836());
            if (compare == 0) {
                return a.method_12832().compareTo(b.method_12832());
            }
            return compare;
        });

        this.codeEditor = new CodeEditor(UPLOAD);
        this.codeEditor.setSaveCallback((source, errorConsumer) -> {
            if (this.selectedProgram == null || !glIsShader(this.editShaderId)) {
                errorConsumer.accept(0, "Invalid Shader");
                return;
            }

            GlStateManager.glShaderSource(this.editShaderId, List.of(source));
            glCompileShader(this.editShaderId);
            if (glGetShaderi(this.editShaderId, GL_COMPILE_STATUS) != GL_TRUE) {
                String log = glGetShaderInfoLog(this.editShaderId);
                parseErrors(source, log).forEach(errorConsumer);
                System.out.println(log);
                return;
            }

            glLinkProgram(this.editProgramId);
            if (glGetProgrami(this.editProgramId, GL_LINK_STATUS) != GL_TRUE) {
                String log = glGetProgramInfoLog(this.editProgramId);
                parseErrors(source, log).forEach(errorConsumer);
                System.out.println(log);
            }
        });
        this.codeEditor.getEditor().setLanguageDefinition(VeilLanguageDefinitions.glsl());

        this.programFilterText = new ImString(128);
        this.programFilter = null;
        this.selectedProgram = null;
        this.selectedTab = 0;

        this.addDefinitionText = new ImString(128);
        this.removedDefinitions = new HashSet<>(1);

        this.editSourceOpen = new ImBoolean();
        this.editProgramId = 0;
        this.editShaderId = 0;
    }

    private void setSelectedProgram(@Nullable class_2960 name) {
        if (name != null && this.shaders.containsKey(name)) {
            int program = this.shaders.getInt(name);
            if (glIsProgram(program)) {
                int[] attachedShaders = new int[glGetProgrami(program, GL_ATTACHED_SHADERS)];
                glGetAttachedShaders(program, null, attachedShaders);

                Map<Integer, Integer> shaders = new Int2IntArrayMap(attachedShaders.length);
                for (int shader : attachedShaders) {
                    shaders.put(glGetShaderi(shader, GL_SHADER_TYPE), shader);
                }

                this.selectedProgram = new SelectedProgram(name, program, Collections.unmodifiableMap(shaders));
                return;
            } else {
                Veil.LOGGER.error("Compiled shader does not exist for program: {}", name);
            }
        }

        this.selectedProgram = null;
    }

    private void setEditShaderSource(int program, int shader) {
        this.editSourceOpen.set(true);
        this.editProgramId = program;
        this.editShaderId = shader;
        this.codeEditor.show(null, glGetShaderSource(shader));
    }

    private static Map<Integer, String> parseErrors(String source, String log) {
        Map<Integer, Map<Integer, String>> logErrors = parseLogErrors(log);

        Map<Integer, String> foundErrors = new HashMap<>();
        int sourceId = 0;
        int lineNumber = 0;
        int sourceLineNumber = -1;
        for (String line : source.split("\n")) {
            sourceLineNumber++;

            Matcher matcher = LINE_DIRECTIVE_PARSER.matcher(line);
            if (matcher.find()) {
                try {
                    lineNumber = Integer.parseInt(matcher.group(1));
                    if (matcher.groupCount() > 1) {
                        sourceId = Integer.parseInt(matcher.group(2));
                    }
                } catch (Throwable ignored) {
                }
                continue;
            }

            Map<Integer, String> errors = logErrors.get(sourceId);
            if (errors != null) {
                String error = errors.remove(lineNumber);
                if (error != null) {
                    foundErrors.put(sourceLineNumber, error);
                }
            }

            lineNumber++;
        }

        return foundErrors;
    }

    private static Map<Integer, Map<Integer, String>> parseLogErrors(String log) {
        Map<Integer, Map<Integer, String>> logErrors = new HashMap<>();
        for (String line : log.split("\n")) {
            Matcher matcher = ERROR_PARSER.matcher(line);
            if (!matcher.find()) {
                continue;
            }

            try {
                int sourceNumber = Integer.parseInt(matcher.group(1));
                int lineNumber = Integer.parseInt(matcher.group(2));

                Map<Integer, String> errors = logErrors.computeIfAbsent(sourceNumber, unused -> new HashMap<>());
                if (errors.containsKey(lineNumber)) {
                    continue;
                }

                String error = matcher.group(3);
                errors.put(lineNumber, error);
            } catch (Throwable ignored) {
            }
        }
        return logErrors;
    }

    private void reloadShaders() {
        this.shaders.clear();
        TabSource.values()[this.selectedTab].addShaders(this.shaders::put);
        if (this.selectedProgram != null && !this.shaders.containsKey(this.selectedProgram.name)) {
            this.setSelectedProgram(null);
        }
    }

    @Override
    public class_2561 getDisplayName() {
        return TITLE;
    }

    @Override
    public class_2561 getGroup() {
        return RENDERER_GROUP;
    }

    @Override
    protected void renderComponents() {
        this.removedDefinitions.clear();

        ImGui.beginChild("##shader_programs", ImGui.getContentRegionAvailX() * 2 / 3, 0);
        VeilImGuiUtil.component(SHADER_PROGRAMS);

        TabSource[] sources = TabSource.values();
        if (ImGui.beginTabBar("##controls")) {
            if (ImGui.tabItemButton(REFRESH.getString())) {
                this.reloadShaders();
            }
            for (TabSource source : sources) {
                if (!source.visible.getAsBoolean()) {
                    continue;
                }

                ImGui.beginDisabled(!source.active.getAsBoolean());
                if (ImGui.beginTabItem(source.displayName.getString())) {
                    if (this.selectedTab != source.ordinal()) {
                        this.selectedTab = source.ordinal();
                        this.setSelectedProgram(null);
                        this.reloadShaders();
                    }
                    ImGui.endTabItem();
                }
                ImGui.endDisabled();
            }
            ImGui.endTabBar();
        }

        // Deselect the tab if it is no longer active
        while (!sources[this.selectedTab].active.getAsBoolean() && this.selectedTab > 0) {
            this.selectedTab--;
            this.setSelectedProgram(null);
            this.reloadShaders();
        }

        ImGui.setNextItemWidth(ImGui.getContentRegionAvailX());
        if (ImGui.inputTextWithHint("##search", SEARCH.getString(), this.programFilterText)) {
            String regex = this.programFilterText.get();
            this.programFilter = null;
            if (!regex.isBlank()) {
                try {
                    this.programFilter = Pattern.compile(regex);
                } catch (PatternSyntaxException ignored) {
                }
            }
        }

        if (ImGui.beginListBox("##programs", ImGui.getContentRegionAvailX(), -Float.MIN_VALUE)) {
            for (Object2IntMap.Entry<class_2960> entry : this.shaders.object2IntEntrySet()) {
                class_2960 name = entry.getKey();
                boolean selected = this.selectedProgram != null && name.equals(this.selectedProgram.name);

                if (this.programFilter != null && !this.programFilter.matcher(name.toString()).find()) {
                    if (selected) {
                        this.setSelectedProgram(null);
                    }
                    continue;
                }

                if (ImGui.selectable("##" + name.toString(), selected)) {
                    this.setSelectedProgram(name);
                }

                ImGui.sameLine();
                VeilImGuiUtil.resourceLocation(name);

                ImGui.pushStyleVar(ImGuiStyleVar.ItemSpacing, 0, ImGui.getStyle().getItemSpacingY());
                ImGui.sameLine();
                ImGui.text(" (" + entry.getIntValue() + ")");
                ImGui.popStyleVar();
            }

            ImGui.endListBox();
        }
        ImGui.endChild();

        ShaderPreDefinitions definitions = VeilRenderSystem.renderer().getShaderDefinitions();
        ImGui.sameLine();
        if (ImGui.beginChild("##panel", 0, ImGui.getContentRegionAvailY())) {
            if (ImGui.beginChild("##open_source", 0, ImGui.getContentRegionAvailY() / 2)) {
                VeilImGuiUtil.component(OPEN_SOURCE);

                this.openShaderButton(GL_FRAGMENT_SHADER);
                this.openShaderButton(GL_VERTEX_SHADER);
                this.openShaderButton(GL_COMPUTE_SHADER);
                this.openShaderButton(GL_GEOMETRY_SHADER);
                this.openShaderButton(GL_TESS_CONTROL_SHADER);
                this.openShaderButton(GL_TESS_EVALUATION_SHADER);
            }
            ImGui.endChild();

            if (ImGui.beginChild("##shader_definitions", 0, ImGui.getContentRegionAvailY())) {
                VeilImGuiUtil.component(SHADER_DEFINITIONS);
                ImGui.setNextItemWidth(ImGui.getContentRegionAvailX());
                if (ImGui.inputTextWithHint("##add_definition", SHADER_DEFINITIONS_HINT.getString(), this.addDefinitionText, ImGuiInputTextFlags.EnterReturnsTrue)) {
                    definitions.define(this.addDefinitionText.get().trim());
                    this.addDefinitionText.clear();
                }
                if (ImGui.beginListBox("##definitions", -Float.MIN_VALUE, ImGui.getContentRegionAvailY())) {
                    for (Map.Entry<String, String> entry : definitions.getDefinitions().entrySet()) {
                        String name = entry.getKey();
                        String value = entry.getValue();

                        ImGui.pushID(name);
                        ImGui.text(value);

                        float size = ImGui.getTextLineHeightWithSpacing();
                        ImGui.sameLine();
                        ImGui.dummy(ImGui.getContentRegionAvailX() - ImGui.getStyle().getCellPaddingX() * 2 - size, 0);
                        ImGui.sameLine();
                        if (ImGui.button("X", size, size)) {
                            this.removedDefinitions.add(name);
                        }

                        ImGui.popID();
                    }

                    ImGui.endListBox();
                }
            }
            ImGui.endChild();
        }
        ImGui.endChild();

        for (String name : this.removedDefinitions) {
            definitions.remove(name);
        }
    }

    @Override
    public void render() {
        ImGui.setNextWindowSizeConstraints(600, 400, Float.MAX_VALUE, Float.MAX_VALUE);

        super.render();

        this.codeEditor.renderWindow();
    }

    private void openShaderButton(int type) {
        boolean disabled = this.selectedProgram == null || !this.selectedProgram.shaders.containsKey(type);
        ImGui.beginDisabled(disabled);

        if (disabled) {
            ImGui.pushStyleColor(ImGuiCol.Button, ImGui.getColorU32(ImGuiCol.FrameBg));
        }

        if (ImGui.button(DeviceInfoViewer.getShaderName(type).getString())) {
            this.setEditShaderSource(this.selectedProgram.programId, this.selectedProgram.shaders.get(type));
        }

        if (disabled) {
            ImGui.popStyleColor();
        }

        ImGui.endDisabled();
    }

    @Override
    public void onShow() {
        super.onShow();
        this.reloadShaders();
    }

    @Override
    public void onHide() {
        super.onHide();
        this.shaders.clear();
    }

    @Override
    public void free() {
        super.free();
        this.codeEditor.free();
    }

    @Override
    public void method_14491(@NotNull class_3300 resourceManager) {
        if (this.isOpen()) {
            this.reloadShaders();
        }
    }

    private record SelectedProgram(class_2960 name, int programId, Map<Integer, Integer> shaders) {
    }

    private enum TabSource {
        VANILLA(class_2561.method_43471("editor.veil.shader.source.vanilla")) {
            @Override
            public void addShaders(ObjIntConsumer<class_2960> registry) {
                GameRendererAccessor gameRenderer = (GameRendererAccessor) class_310.method_1551().field_1773;
                Map<String, class_5944> shaders = gameRenderer.getShaders();
                for (class_5944 shader : shaders.values()) {
                    String name = shader.method_35787().isBlank() ? Integer.toString(shader.method_1270()) : shader.method_35787();
                    registry.accept(class_2960.method_60654(name), shader.method_1270());
                }

                class_5944 blitShader = gameRenderer.getBlitShader();
                registry.accept(class_2960.method_60654(blitShader.method_35787()), blitShader.method_1270());
            }
        },
        VANILLA_POST(class_2561.method_43471("editor.veil.shader.source.vanilla_post")) {
            @Override
            public void addShaders(ObjIntConsumer<class_2960> registry) {
                LevelRendererAccessor levelRenderer = (LevelRendererAccessor) class_310.method_1551().field_1769;
                this.addChainPasses(registry, levelRenderer.getEntityEffect());
                this.addChainPasses(registry, levelRenderer.getTransparencyChain());
                GameRendererAccessor gameRenderer = (GameRendererAccessor) class_310.method_1551().field_1773;
                this.addChainPasses(registry, gameRenderer.getPostEffect());
            }

            private void addChainPasses(ObjIntConsumer<class_2960> registry, @Nullable class_279 chain) {
                if (chain == null) {
                    return;
                }

                List<class_283> passes = ((PostChainAccessor) chain).getPasses();
                for (class_283 pass : passes) {
                    class_280 effect = pass.method_1295();
                    registry.accept(class_2960.method_60654(effect.method_35763()), effect.method_1270());
                }
            }
        },
        VEIL(class_2561.method_43471("editor.veil.shader.source.veil")) {
            @Override
            public void addShaders(ObjIntConsumer<class_2960> registry) {
                Map<class_2960, ShaderProgram> shaders = VeilRenderSystem.renderer().getShaderManager().getShaders();
                for (ShaderProgram shader : shaders.values()) {
                    registry.accept(shader.getId(), shader.getProgram());
                }
                VeilImGuiImpl.get().addImguiShaders(registry);
            }
        },
        VEIL_DEFERRED(class_2561.method_43471("editor.veil.shader.source.veil_deferred"), VeilRenderSystem.renderer().getDeferredRenderer()::isEnabled, () -> true) {
            @Override
            public void addShaders(ObjIntConsumer<class_2960> registry) {
                Map<class_2960, ShaderProgram> shaders = VeilRenderSystem.renderer().getDeferredRenderer().getDeferredShaderManager().getShaders();
                for (ShaderProgram shader : shaders.values()) {
                    registry.accept(shader.getId(), shader.getProgram());
                }
            }
        },
        IRIS(class_2561.method_43471("editor.veil.shader.source.iris"), IrisShaderMap::isEnabled, IrisShaderMap::isEnabled) {
            @Override
            public void addShaders(ObjIntConsumer<class_2960> registry) {
                for (class_5944 shader : IrisShaderMap.getLoadedShaders()) {
                    String name = shader.method_35787().isBlank() ? Integer.toString(shader.method_1270()) : shader.method_35787();
                    registry.accept(class_2960.method_60654(name), shader.method_1270());
                }
            }
        },
        SODIUM(class_2561.method_43471("editor.veil.shader.source.sodium"), SodiumShaderMap::isEnabled, SodiumShaderMap::isEnabled) {
            @Override
            public void addShaders(ObjIntConsumer<class_2960> registry) {
                for (Object2IntMap.Entry<class_2960> entry : SodiumShaderMap.getLoadedShaders().object2IntEntrySet()) {
                    registry.accept(entry.getKey(), entry.getIntValue());
                }
            }
        },
        OTHER(class_2561.method_43471("editor.veil.shader.source.unknown")) {
            @Override
            public void addShaders(ObjIntConsumer<class_2960> registry) {
                IntSet programs = new IntOpenHashSet();
                for (int i = 1; i < 10000; i++) {
                    if (!glIsProgram(i)) {
                        continue;
                    }

                    programs.add(i);
                }

                for (TabSource value : TabSource.values()) {
                    if (value == this) {
                        continue;
                    }

                    value.addShaders((name, id) -> programs.remove(id));
                }

                for (int program : programs) {
                    registry.accept(class_2960.method_60655("unknown", Integer.toString(program)), program);
                }
            }
        };

        private final class_2561 displayName;
        private final BooleanSupplier active;
        private final BooleanSupplier visible;

        TabSource(class_2561 displayName) {
            this(displayName, () -> true, () -> true);
        }

        TabSource(class_2561 displayName, BooleanSupplier active, BooleanSupplier visible) {
            this.displayName = displayName;
            this.active = active;
            this.visible = visible;
        }

        public abstract void addShaders(ObjIntConsumer<class_2960> registry);
    }
}
