package foundry.veil.impl.client.render.rendertype;

import com.google.gson.JsonElement;
import com.google.gson.JsonParser;
import com.google.gson.JsonSyntaxException;
import com.mojang.serialization.DataResult;
import com.mojang.serialization.JsonOps;
import foundry.veil.Veil;
import foundry.veil.api.client.render.rendertype.layer.CompositeRenderTypeData;
import it.unimi.dsi.fastutil.ints.Int2ObjectArrayMap;
import it.unimi.dsi.fastutil.ints.Int2ObjectMap;
import it.unimi.dsi.fastutil.objects.Object2ObjectArrayMap;
import org.jetbrains.annotations.ApiStatus;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;
import net.minecraft.class_1921;
import net.minecraft.class_2960;
import net.minecraft.class_3298;
import net.minecraft.class_3300;
import net.minecraft.class_3695;
import net.minecraft.class_4080;
import net.minecraft.class_7654;

@ApiStatus.Internal
public class DynamicRenderTypeManager extends class_4080<Map<class_2960, byte[]>> {

    private static final class_7654 CONVERTER = class_7654.method_45114("pinwheel/rendertypes");

    private final Map<class_2960, RenderTypeCache> renderTypes = new Object2ObjectArrayMap<>();

    public @Nullable class_1921 get(class_2960 id, Object... params) {
        RenderTypeCache cache = this.renderTypes.get(id);
        if (cache == null) {
            return null;
        }

        return cache.get(params);
    }

    @Override
    protected @NotNull Map<class_2960, byte[]> method_18789(@NotNull class_3300 resourceManager, @NotNull class_3695 profilerFiller) {
        Map<class_2960, byte[]> data = new HashMap<>();

        Map<class_2960, class_3298> resources = CONVERTER.method_45113(resourceManager);
        for (Map.Entry<class_2960, class_3298> entry : resources.entrySet()) {
            class_2960 location = entry.getKey();
            class_2960 id = CONVERTER.method_45115(location);

            try (InputStream stream = entry.getValue().method_14482()) {
                data.put(id, stream.readAllBytes());
            } catch (Exception e) {
                Veil.LOGGER.error("Couldn't read data file {} from {}", id, location, e);
            }
        }

        return data;
    }

    @Override
    protected void apply(Map<class_2960, byte[]> fileData, class_3300 resourceManager, class_3695 profilerFiller) {
        Map<class_2960, RenderTypeCache> renderTypes = new HashMap<>();

        for (Map.Entry<class_2960, byte[]> entry : fileData.entrySet()) {
            class_2960 id = entry.getKey();

            try (Reader reader = new InputStreamReader(new ByteArrayInputStream(entry.getValue()))) {
                JsonElement element = JsonParser.parseReader(reader);
                DataResult<CompositeRenderTypeData> result = CompositeRenderTypeData.CODEC.parse(JsonOps.INSTANCE, element);

                if (result.error().isPresent()) {
                    throw new JsonSyntaxException(result.error().get().message());
                }

                CompositeRenderTypeData data = result.result().orElseThrow();
                if (renderTypes.put(id, new RenderTypeCache(id.toString(), data)) != null) {
                    throw new IllegalStateException("Duplicate data file ignored with ID " + id);
                }
            } catch (Exception e) {
                Veil.LOGGER.error("Couldn't parse data file {} from {}", id, CONVERTER.method_45112(id), e);
            }
        }

        this.renderTypes.clear();
        this.renderTypes.putAll(renderTypes);
        Veil.LOGGER.info("Loaded {} render types", renderTypes.size());
    }

    private static class RenderTypeCache {

        private final String name;
        private final CompositeRenderTypeData data;
        private final Int2ObjectMap<class_1921> objectCache;
        private class_1921 defaultCache;
        private boolean defaultError;

        public RenderTypeCache(String name, CompositeRenderTypeData data) {
            this.name = name;
            this.data = data;
            this.objectCache = new Int2ObjectArrayMap<>(4);
            this.defaultCache = null;
        }

        public @Nullable class_1921 get(Object... params) {
            if (params.length == 0) {
                if (this.defaultError) {
                    return null;
                }

                if (this.defaultCache == null) {
                    try {
                        this.defaultCache = this.data.createRenderType(this.name);
                    } catch (Exception e) {
                        Veil.LOGGER.error("Failed to create rendertype {} with no parameters", this.name, e);
                        this.defaultError = true;
                    }
                }
                return this.defaultCache;
            }

            return this.objectCache.computeIfAbsent(Arrays.hashCode(params), i -> {
                try {
                    return this.data.createRenderType(this.name, params);
                } catch (Exception e) {
                    Veil.LOGGER.error("Failed to create rendertype {} with parameters: [{}]", this.name, Arrays.stream(params).map(Objects::toString).collect(Collectors.joining(", ")), e);
                    return null;
                }
            });
        }
    }
}
