package foundry.veil.api.client.render;

import foundry.veil.api.client.render.framebuffer.AdvancedFbo;
import foundry.veil.api.client.render.rendertype.VeilRenderTypeBuilder;
import foundry.veil.api.client.render.shader.ShaderManager;
import foundry.veil.api.client.render.shader.program.ShaderProgram;
import foundry.veil.impl.client.render.pipeline.AdvancedFboShard;
import foundry.veil.impl.client.render.pipeline.PatchStateShard;
import foundry.veil.impl.client.render.pipeline.ShaderProgramShard;
import foundry.veil.impl.client.render.shader.program.ShaderProgramImpl;
import foundry.veil.impl.client.render.wrapper.DSAVanillaAdvancedFboWrapper;
import foundry.veil.impl.client.render.wrapper.LegacyVanillaAdvancedFboWrapper;
import java.util.function.Supplier;
import net.minecraft.class_1921;
import net.minecraft.class_276;
import net.minecraft.class_284;
import net.minecraft.class_2960;
import net.minecraft.class_4587;
import net.minecraft.class_4604;
import net.minecraft.class_4668;
import net.minecraft.class_5944;

/**
 * Bridges between Minecraft and Veil render classes.
 *
 * @author Ocelot
 */
public interface VeilRenderBridge {

    /**
     * <p>Wraps the specified shader with a vanilla Minecraft shader instance wrapper. There are a few special properties about the shader wrapper.</p>
     * <ul>
     *     <li>The shader instance cannot be used to free the shader program. {@link ShaderProgram#free()} must be called separately.
     *     If the shader is loaded through {@link ShaderManager} then there is no need to free the shader.</li>
     *     <li>Calling {@link class_284#method_1300()} will do nothing since the values are uploaded when the appropriate methods are called</li>
     *     <li>Uniforms are lazily wrapped and will not crash when the wrong method is called.</li>
     *     <li>{@link class_284#method_35659(int, float)} is not supported and will throw an {@link UnsupportedOperationException}.</li>
     *     <li>{@link class_284#method_1253(float[])} only works for 1, 2, 3, and 4 float elements. Any other size will throw an {@link UnsupportedOperationException}.</li>
     * </ul>
     *
     * @param program The program to create a shader instance from
     * @return A lazily loaded shader instance wrapper for this program
     */
    static class_5944 toShaderInstance(ShaderProgram program) {
        return ((ShaderProgramImpl) program).toShaderInstance();
    }

    /**
     * Creates a cull frustum helper from the specified vanilla frustum.
     *
     * @param frustum The frustum to use for the cull frustum
     * @return The cull frustum
     */
    static CullFrustum create(class_4604 frustum) {
        return (CullFrustum) frustum;
    }

    /**
     * Creates a render type builder helper from the specified vanilla composite state builder.
     *
     * @param builder The state builder to wrap
     * @return The render type builder
     */
    static VeilRenderTypeBuilder create(class_1921.class_4688.class_4689 builder) {
        return (VeilRenderTypeBuilder) builder;
    }

    /**
     * Creates a matrix stack wrapper for the specified post stack.
     *
     * @param poseStack The pose stack to wrap
     * @return The matrix stack representation
     */
    static MatrixStack create(class_4587 poseStack) {
        return (MatrixStack) poseStack;
    }

    /**
     * Wraps the specified render target in a new advanced fbo.
     *
     * @param renderTarget The render target instance
     * @return A new advanced fbo that wraps the target in the api
     */
    static AdvancedFbo wrap(class_276 renderTarget) {
        return VeilRenderBridge.wrap(() -> renderTarget);
    }

    /**
     * Wraps the specified render target in a new advanced fbo.
     *
     * @param renderTargetSupplier The supplier to the render target instance
     * @return A new advanced fbo that wraps the target in the api
     */
    static AdvancedFbo wrap(Supplier<class_276> renderTargetSupplier) {
        return VeilRenderSystem.directStateAccessSupported() ? new DSAVanillaAdvancedFboWrapper(renderTargetSupplier) : new LegacyVanillaAdvancedFboWrapper(renderTargetSupplier);
    }

    /**
     * Creates a new shader state that points to the specified Veil shader name.
     *
     * @param shader The name of the shader to point to.
     * @return A new shader state shard for that shader
     */
    static class_4668.class_5942 shaderState(class_2960 shader) {
        return new ShaderProgramShard(shader);
    }

    /**
     * Creates a new output state that draws into the specified Veil framebuffer.
     *
     * @param framebuffer The framebuffer to use
     * @return A new shader state shard for that shader
     */
    static class_4668.class_4678 outputState(class_2960 framebuffer) {
        return new AdvancedFboShard(framebuffer, () -> VeilRenderSystem.renderer().getFramebufferManager().getFramebuffer(framebuffer));
    }

    /**
     * Creates a new output state that draws into the specified Veil framebuffer.
     *
     * @param framebuffer The framebuffer to use
     * @return A new shader state shard for that shader
     */
    static class_4668.class_4678 outputState(AdvancedFbo framebuffer) {
        return new AdvancedFboShard(null, () -> framebuffer);
    }

    /**
     * Creates a new output state that draws into the specified Veil framebuffer.
     *
     * @param framebuffer A supplier to the framebuffer to use
     * @return A new shader state shard for that shader
     */
    static class_4668.class_4678 outputState(Supplier<AdvancedFbo> framebuffer) {
        return new AdvancedFboShard(null, framebuffer);
    }

    /**
     * Creates a new render state shard for tesselation patch size.
     *
     * @param patchVertices The number of vertices per patch
     * @return A new patch state
     */
    static class_4668 patchState(int patchVertices) {
        return new PatchStateShard(patchVertices);
    }
}
