package foundry.veil.impl.client.render.perspective;

import it.unimi.dsi.fastutil.ints.IntArraySet;
import it.unimi.dsi.fastutil.ints.IntSet;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.joml.Vector3d;

import java.util.*;
import java.util.function.Consumer;
import net.minecraft.class_2338;
import net.minecraft.class_2350;
import net.minecraft.class_243;
import net.minecraft.class_3532;
import net.minecraft.class_4076;
import net.minecraft.class_4184;
import net.minecraft.class_4604;
import net.minecraft.class_5539;
import net.minecraft.class_769;
import net.minecraft.class_846;
import net.minecraft.class_8603;

public class VeilSectionOcclusionGraph {

    private static final class_2350[] DIRECTIONS = class_2350.values();
    private static final int MINIMUM_ADVANCED_CULLING_DISTANCE = 60;
    private static final double CEILED_SECTION_DIAGONAL = Math.ceil(Math.sqrt(3.0) * 16.0);

    private final NodeQueue nodeQueue = new NodeQueue(64);
    private class_769 viewArea;
    private int viewDistance;

    public void update(class_769 viewArea, boolean smartCull, LevelPerspectiveCamera camera, class_4604 frustum, List<class_846.class_851> sections) {
        this.viewArea = viewArea;
        this.viewDistance = Math.min(viewArea.method_52839(), class_3532.method_15386(camera.getRenderDistance()));
        GraphStorage graphState = new GraphStorage(viewArea.field_4150.length);
        this.initializeQueueForFullUpdate(camera, viewArea);
        this.nodeQueue.forEach(node -> graphState.sectionToNodeMap.put(node.section, node));
        this.runUpdates(graphState, viewArea, camera.method_19326(), frustum, this.nodeQueue, smartCull, sections);
    }

    public void reset() {
        this.nodeQueue.trim(64);
    }

    private void initializeQueueForFullUpdate(class_4184 camera, class_769 viewArea) {
        this.nodeQueue.clear();

        class_2338 pos = camera.method_19328();
        class_846.class_851 renderSection = viewArea.method_3323(pos);
        if (renderSection != null) {
            this.nodeQueue.add(new Node(renderSection, 0));
            return;
        }

        class_243 cameraPos = camera.method_19326();
        class_5539 level = viewArea.method_52840();
        boolean aboveVoid = pos.method_10264() > level.method_31607();
        int startY = aboveVoid ? level.method_31600() - 8 : level.method_31607() + 8;
        int startX = class_3532.method_15357(cameraPos.field_1352 / 16.0) << class_4076.field_33096;
        int startZ = class_3532.method_15357(cameraPos.field_1350 / 16.0) << class_4076.field_33096;
        int radius = this.viewDistance;
        List<Node> list = new ArrayList<>(4 * radius * radius);

        class_2338.class_2339 renderSectionPos = new class_2338.class_2339();
        for (int x = -radius; x <= radius; x++) {
            for (int z = -radius; z <= radius; z++) {
                class_846.class_851 section = viewArea.method_3323(renderSectionPos.method_10103(startX + x << class_4076.field_33096 + 8, startY, startZ + z << class_4076.field_33096 + 8));
                if (section != null && this.isInViewDistance(pos, section.method_3670())) {
                    class_2350 direction = aboveVoid ? class_2350.field_11033 : class_2350.field_11036;
                    Node node = new Node(section, 0);
                    node.addSourceDirection(direction);
                    node.addDirection(direction);
                    if (x > 0) {
                        node.addDirection(class_2350.field_11034);
                    } else if (x < 0) {
                        node.addDirection(class_2350.field_11039);
                    }

                    if (z > 0) {
                        node.addDirection(class_2350.field_11035);
                    } else if (z < 0) {
                        node.addDirection(class_2350.field_11043);
                    }

                    list.add(node);
                }
            }
        }

        list.sort(Comparator.comparingDouble(node -> {
            class_2338 origin = node.section.method_3670();
            return pos.method_10268(origin.method_10263() + 8.5, origin.method_10264() + 8.5, origin.method_10260() + 8.5);
        }));

        this.nodeQueue.addAll(list);
    }

    private void runUpdates(
            GraphStorage graphStorage,
            class_769 viewArea,
            class_243 cameraPosition,
            class_4604 frustum,
            Queue<Node> nodeQueue,
            boolean smartCull,
            List<class_846.class_851> sections
    ) {
        class_2338 cameraSectionPos = new class_2338(
                class_3532.method_15357(cameraPosition.field_1352 / 16.0) << class_4076.field_33096,
                class_3532.method_15357(cameraPosition.field_1351 / 16.0) << class_4076.field_33096,
                class_3532.method_15357(cameraPosition.field_1350 / 16.0) << class_4076.field_33096);
        class_2338 cameraCenter = cameraSectionPos.method_10069(8, 8, 8);
        class_2338.class_2339 temp = new class_2338.class_2339();

        class_5539 level = viewArea.method_52840();
        int maxBuildHeight = level.method_31600();
        int minBuildHeight = level.method_31607();

        while (!nodeQueue.isEmpty()) {
            Node node = nodeQueue.poll();
            class_846.class_851 renderSection = node.section;
            if (frustum.method_23093(renderSection.method_40051()) && graphStorage.renderSections.add(renderSection.field_29641)) {
                sections.add(node.section);
            }

            class_2338 origin = renderSection.method_3670();
            boolean far = Math.abs(origin.method_10263() - cameraSectionPos.method_10263()) > MINIMUM_ADVANCED_CULLING_DISTANCE
                    || Math.abs(origin.method_10264() - cameraSectionPos.method_10264()) > MINIMUM_ADVANCED_CULLING_DISTANCE
                    || Math.abs(origin.method_10260() - cameraSectionPos.method_10260()) > MINIMUM_ADVANCED_CULLING_DISTANCE;

            for (class_2350 direction : DIRECTIONS) {
                class_846.class_851 section = this.getRelativeFrom(cameraSectionPos, renderSection, direction);
                if (section == null) {
                    continue;
                }

                class_2350 opposite = direction.method_10153();
                if (!smartCull || (node.directions & (1 << opposite.ordinal())) == 0) {
                    if (smartCull && node.sourceDirections != 0) {
                        class_846.class_849 compiledSection = renderSection.method_3677();
                        boolean visible = false;

                        for (int i = 0; i < DIRECTIONS.length; i++) {
                            if ((node.sourceDirections & (1 << i)) != 0 && compiledSection.method_3650(DIRECTIONS[i], opposite)) {
                                visible = true;
                                break;
                            }
                        }

                        if (!visible) {
                            continue;
                        }
                    }

                    class_2338 neighborOrigin = section.method_3670();
                    if (smartCull && far) {
                        // TODO this looks like voxel ray marching
                        int offsetX = (direction.method_10166() == class_2350.class_2351.field_11048 ? cameraCenter.method_10263() <= neighborOrigin.method_10263() : cameraCenter.method_10263() >= neighborOrigin.method_10263()) ? 0 : 16;
                        int offsetY = (direction.method_10166() == class_2350.class_2351.field_11052 ? cameraCenter.method_10264() <= neighborOrigin.method_10264() : cameraCenter.method_10264() >= neighborOrigin.method_10264()) ? 0 : 16;
                        int offsetZ = (direction.method_10166() == class_2350.class_2351.field_11051 ? cameraCenter.method_10260() <= neighborOrigin.method_10260() : cameraCenter.method_10260() >= neighborOrigin.method_10260()) ? 0 : 16;
                        temp.method_25504(neighborOrigin, offsetX, offsetY, offsetZ);
                        Vector3d pos = new Vector3d(cameraPosition.field_1352 - temp.method_10263(), cameraPosition.field_1351 - temp.method_10264(), cameraPosition.field_1350 - temp.method_10260());
                        Vector3d step = pos.normalize(CEILED_SECTION_DIAGONAL, new Vector3d());
                        boolean visible = true;

                        while (pos.distanceSquared(cameraPosition.field_1352, cameraPosition.field_1351, cameraPosition.field_1350) > MINIMUM_ADVANCED_CULLING_DISTANCE * MINIMUM_ADVANCED_CULLING_DISTANCE) {
                            pos.add(step);
                            if (pos.y > maxBuildHeight || pos.y < minBuildHeight) {
                                break;
                            }

                            class_846.class_851 renderSection3 = viewArea.method_3323(temp.method_10102(Math.floor(pos.x), Math.floor(pos.y), Math.floor(pos.z)));
                            if (renderSection3 == null || graphStorage.sectionToNodeMap.get(renderSection3) == null) {
                                visible = false;
                                break;
                            }
                        }

                        if (!visible) {
                            continue;
                        }
                    }

                    Node node2 = graphStorage.sectionToNodeMap.get(section);
                    if (node2 != null) {
                        node2.addSourceDirection(direction);
                    } else {
                        Node node3 = new Node(section, node.step + 1);
                        node3.addSourceDirection(direction);
                        node3.addDirection(direction);
                        if (section.method_3673()) {
                            nodeQueue.add(node3);
                            graphStorage.sectionToNodeMap.put(section, node3);
                        } else if (this.isInViewDistance(cameraSectionPos, neighborOrigin)) {
                            graphStorage.sectionToNodeMap.put(section, node3);
                        }
                    }
                }
            }
        }
    }

    private boolean isInViewDistance(class_2338 pos, class_2338 origin) {
        int centerX = pos.method_10263() >> class_4076.field_33096;
        int centerZ = pos.method_10260() >> class_4076.field_33096;
        int x = origin.method_10263() >> class_4076.field_33096;
        int z = origin.method_10260() >> class_4076.field_33096;
        return class_8603.method_52358(centerX, centerZ, this.viewDistance, x, z, false);
    }

    @Nullable
    private class_846.class_851 getRelativeFrom(class_2338 pos, class_846.class_851 section, class_2350 direction) {
        class_2338 origin = section.method_3676(direction);
        if (!this.isInViewDistance(pos, origin)) {
            return null;
        } else {
            return class_3532.method_15382(pos.method_10264() - origin.method_10264()) > this.viewDistance << class_4076.field_33096 ? null : this.viewArea.method_3323(origin);
        }
    }

    private static class GraphStorage {
        public final SectionToNodeMap sectionToNodeMap;
        public final IntSet renderSections;

        public GraphStorage(int size) {
            this.sectionToNodeMap = new SectionToNodeMap(size);
            this.renderSections = new IntArraySet(size);
        }
    }

    private static class Node {

        private final class_846.class_851 section;
        private int sourceDirections;
        private int directions;
        private final int step;

        private Node(class_846.class_851 section, int step) {
            this.section = section;
            this.step = step;
        }

        private void addDirection(class_2350 direction) {
            this.directions |= 1 << direction.ordinal();
        }

        private void addSourceDirection(class_2350 sourceDirection) {
            this.sourceDirections |= 1 << sourceDirection.ordinal();
        }

        @Override
        public int hashCode() {
            return this.section.method_3670().hashCode();
        }

        @Override
        public boolean equals(Object object) {
            return this.section.method_3670().equals(((Node) object).section.method_3670());
        }
    }

    private static class SectionToNodeMap {
        private final Node[] nodes;

        private SectionToNodeMap(int size) {
            this.nodes = new Node[size];
        }

        public void put(class_846.class_851 section, Node node) {
            this.nodes[section.field_29641] = node;
        }

        public @Nullable Node get(class_846.class_851 section) {
            int index = section.field_29641;
            return index >= 0 && index < this.nodes.length ? this.nodes[index] : null;
        }
    }

    private static class NodeQueue implements Queue<Node> {

        private Node[] data;
        private int size;
        private int readPointer;

        public NodeQueue(int initialCapacity) {
            this.data = new Node[initialCapacity];
            this.size = 0;
        }

        public void trim(int size) {
            if (this.data.length > size) {
                this.data = new Node[size];
            }
            this.size = 0;
            this.readPointer = 0;
        }

        @Override
        public int size() {
            return this.size;
        }

        @Override
        public boolean isEmpty() {
            return this.readPointer >= this.size;
        }

        @Override
        public boolean contains(Object o) {
            throw new UnsupportedOperationException();
        }

        @Override
        public @NotNull Iterator<Node> iterator() {
            throw new UnsupportedOperationException();
        }

        @Override
        public @NotNull Object[] toArray() {
            throw new UnsupportedOperationException();
        }

        @Override
        public @NotNull <T> T[] toArray(@NotNull T[] a) {
            throw new UnsupportedOperationException();
        }

        @Override
        public boolean add(Node node) {
            if (this.size >= this.data.length) {
                this.data = Arrays.copyOf(this.data, this.data.length * 2);
            }
            this.data[this.size++] = node;
            return true;
        }

        @Override
        public boolean remove(Object o) {
            throw new UnsupportedOperationException();
        }

        @Override
        public boolean containsAll(@NotNull Collection<?> c) {
            throw new UnsupportedOperationException();
        }

        @Override
        public boolean addAll(@NotNull Collection<? extends Node> c) {
            if (this.isEmpty() && c instanceof ArrayList<? extends Node> arrayList) {
                this.data = arrayList.toArray(Node[]::new);
                return true;
            }
            if (this.size + c.size() > this.data.length) {
                this.data = Arrays.copyOf(this.data, this.size + c.size());
            }
            for (Node node : c) {
                this.data[this.size++] = node;
            }
            return true;
        }

        @Override
        public boolean removeAll(@NotNull Collection<?> c) {
            throw new UnsupportedOperationException();
        }

        @Override
        public boolean retainAll(@NotNull Collection<?> c) {
            throw new UnsupportedOperationException();
        }

        @Override
        public void clear() {
            this.size = 0;
            this.readPointer = 0;
        }

        @Override
        public boolean offer(Node node) {
            return this.add(node);
        }

        @Override
        public Node remove() {
            if (this.readPointer < this.size) {
                return this.data[this.readPointer++];
            }
            throw new NoSuchElementException();
        }

        @Override
        public Node poll() {
            if (this.readPointer < this.size) {
                return this.data[this.readPointer++];
            }
            return null;
        }

        @Override
        public Node element() {
            if (this.readPointer < this.size) {
                return this.data[this.readPointer];
            }
            throw new NoSuchElementException();
        }

        @Override
        public Node peek() {
            return this.readPointer >= this.size ? null : this.data[this.readPointer];
        }

        @Override
        public void forEach(Consumer<? super Node> action) {
            for (int i = this.readPointer; i < this.size; i++) {
                action.accept(this.data[i]);
            }
        }
    }
}
