package foundry.veil.api.quasar.particle;

import com.google.common.base.Suppliers;
import foundry.veil.api.TickTaskScheduler;
import foundry.veil.api.quasar.data.ParticleSettings;
import foundry.veil.api.quasar.data.QuasarParticleData;
import foundry.veil.api.quasar.emitters.module.*;
import gg.moonflower.molangcompiler.api.MolangEnvironment;
import gg.moonflower.molangcompiler.api.MolangExpression;
import gg.moonflower.molangcompiler.api.MolangRuntime;
import org.jetbrains.annotations.ApiStatus;
import org.joml.Vector3d;
import org.joml.Vector3f;

import java.util.Iterator;
import java.util.List;
import java.util.function.Supplier;
import net.minecraft.class_1297;
import net.minecraft.class_1309;
import net.minecraft.class_2338;
import net.minecraft.class_238;
import net.minecraft.class_243;
import net.minecraft.class_2680;
import net.minecraft.class_3532;
import net.minecraft.class_5819;
import net.minecraft.class_638;
import net.minecraft.class_761;

public class QuasarParticle {

    private static final double MAXIMUM_COLLISION_VELOCITY_SQUARED = class_3532.method_33723(100.0D);

    private final class_638 level;
    private final class_5819 randomSource;
    private final TickTaskScheduler scheduler;
    private final QuasarParticleData data;
    private final ParticleSettings settings;
    private final ParticleEmitter emitter;
    private final ParticleModuleSet modules;
    private final Vector3d position;
    private final Vector3d velocity;
    private final Vector3f rotation;
    private final class_2338.class_2339 blockPosition;
    private final boolean hasCollision;
    private float radius;
    private final int lifetime;
    private int age;
    private class_238 boundingBox;
    private boolean stoppedByCollision;

    private final Supplier<MolangRuntime> environment;
    private final RenderData renderData;

    public QuasarParticle(class_638 level, class_5819 randomSource, TickTaskScheduler scheduler, QuasarParticleData data, ParticleModuleSet modules, ParticleSettings settings, ParticleEmitter emitter) {
        this.level = level;
        this.randomSource = randomSource;
        this.scheduler = scheduler;
        this.data = data;
        this.settings = settings;
        this.emitter = emitter;
        this.modules = modules;
        this.position = new Vector3d();
        this.velocity = new Vector3d();
        this.rotation = new Vector3f();
        this.blockPosition = new class_2338.class_2339();
        this.hasCollision = this.modules.getCollisionModules().length > 0;
        this.radius = settings.particleSize(this.randomSource);
        this.lifetime = settings.particleLifetime(this.randomSource);
        this.age = 0;

        this.renderData = new RenderData(data);
        // Don't create the environment if the particle never uses it
        this.environment = Suppliers.memoize(() -> MolangRuntime.runtime()
                .setQuery("x", MolangExpression.of(() -> (float) this.renderData.getRenderPosition().x()))
                .setQuery("y", MolangExpression.of(() -> (float) this.renderData.getRenderPosition().y()))
                .setQuery("z", MolangExpression.of(() -> (float) this.renderData.getRenderPosition().z()))
                .setQuery("velX", MolangExpression.of(() -> (float) this.velocity.x()))
                .setQuery("velY", MolangExpression.of(() -> (float) this.velocity.y()))
                .setQuery("velZ", MolangExpression.of(() -> (float) this.velocity.z()))
                .setQuery("speedSq", MolangExpression.of(() -> (float) this.velocity.lengthSquared()))
                .setQuery("speed", MolangExpression.of(() -> (float) this.velocity.length()))
                .setQuery("xRot", MolangExpression.of(() -> (float) Math.toDegrees(this.renderData.getRenderRotation().x())))
                .setQuery("yRot", MolangExpression.of(() -> (float) Math.toDegrees(this.renderData.getRenderRotation().y())))
                .setQuery("zRot", MolangExpression.of(() -> (float) Math.toDegrees(this.renderData.getRenderRotation().z())))
                .setQuery("scale", MolangExpression.of(this.renderData::getRenderRadius))
                .setQuery("age", MolangExpression.of(this.renderData::getRenderAge))
                .setQuery("agePercent", MolangExpression.of(this.renderData::getAgePercent))
                .setQuery("lifetime", this.lifetime)
                .create());
    }

    private void move(double dx, double dy, double dz) {
        if (this.stoppedByCollision || (dx == 0.0D && dy == 0.0D && dz == 0.0D)) {
            return;
        }

        class_238 box = this.getBoundingBox();
        double d0 = dx;
        double d1 = dy;
        double d2 = dz;
        if (this.hasCollision && dx * dx + dy * dy + dz * dz < MAXIMUM_COLLISION_VELOCITY_SQUARED) {
            class_243 vec3 = class_1297.method_20736(null, new class_243(dx, dy, dz), box, this.level, List.of());
            dx = vec3.field_1352;
            dy = vec3.field_1351;
            dz = vec3.field_1350;
        }

        if (dx != 0.0D || dy != 0.0D || dz != 0.0D) {
            this.position.add(dx, dy, dz);
            this.updateBoundingBox();
        }

        if (!this.hasCollision) {
            return;
        }

        List<class_1297> entities = this.level.method_8335(null, box);
        for (class_1297 entity : entities) {
            if (entity instanceof class_1309 livingEntity && livingEntity.method_5805()) {
                this.stoppedByCollision = true;
                break;
            }
        }

        if (Math.abs(d1) >= (double) 1.0E-5F && Math.abs(dy) < (double) 1.0E-5F) {
            this.stoppedByCollision = true;
        }

        if (d0 != dx) {
            this.velocity.x = 0;
            this.stoppedByCollision = true;
        }

        if (d1 != dy) {
            this.velocity.y = 0;
            this.stoppedByCollision = true;
        }

        if (d2 != dz) {
            this.velocity.z = 0;
            this.stoppedByCollision = true;
        }

        // Notify listeners
        if (this.stoppedByCollision) {
            for (CollisionParticleModule collisionParticle : this.modules.getCollisionModules()) {
                collisionParticle.collide(this);
            }
        }
    }

    private void updateBoundingBox() {
        double r = this.radius / 2.0;
        this.boundingBox = new class_238(this.position.x - r, this.position.y - r, this.position.z - r, this.position.x + r, this.position.y + r, this.position.z + r);
    }

    private int getLightColor() {
        return class_761.method_23794(this.level, this.getBlockPosition());
    }

    @ApiStatus.Internal
    public void init() {
        for (InitParticleModule initModule : this.modules.getInitModules()) {
            initModule.init(this);
        }
        int packedLight = this.renderData.getFixedPackedLight();
        this.renderData.tick(this, packedLight == -1 ? this.getLightColor() : packedLight);
        this.updateBoundingBox();
    }

    @ApiStatus.Internal
    public void tick() {
        int packedLight = this.renderData.getFixedPackedLight();
        this.renderData.tick(this, packedLight == -1 ? this.getLightColor() : packedLight);
        this.modules.updateEnabled();
        for (UpdateParticleModule updateModule : this.modules.getUpdateModules()) {
            updateModule.update(this);
        }

        // TODO properly do forces
        for (ForceParticleModule updateModule : this.modules.getForceModules()) {
            updateModule.applyForce(this);
        }

        this.move(this.velocity.x, this.velocity.y, this.velocity.z);

        this.age++;
        if (this.age >= this.lifetime) {
            this.remove();
        }
    }

    @ApiStatus.Internal
    public void render(float partialTicks) {
        Iterator<RenderParticleModule> iterator = this.modules.getEnabledRenderModules();
        while (iterator.hasNext()) {
            iterator.next().render(this, partialTicks);
        }
        this.renderData.render(this, partialTicks);
    }

    @ApiStatus.Internal
    public void onRemove() {
        for (ParticleModule module : this.modules.getAllModules()) {
            module.onRemove();
        }
    }

    public void remove() {
        this.age = Integer.MIN_VALUE;
    }

    public boolean isRemoved() {
        return this.age < 0;
    }

    public class_638 getLevel() {
        return this.level;
    }

    public class_5819 getRandomSource() {
        return this.randomSource;
    }

    public TickTaskScheduler getScheduler() {
        return this.scheduler;
    }

    public QuasarParticleData getData() {
        return this.data;
    }

    public ParticleSettings getSettings() {
        return this.settings;
    }

    public ParticleEmitter getEmitter() {
        return this.emitter;
    }

    public ParticleModuleSet getModules() {
        return this.modules;
    }

    public Vector3d getPosition() {
        return this.position;
    }

    public class_2338 getBlockPosition() {
        return this.blockPosition.method_10102(this.position.x, this.position.y, this.position.z);
    }

    public Vector3d getVelocity() {
        return this.velocity;
    }

    public class_2680 getBlockStateInOrUnder() {
        class_2680 in = this.level.method_8320(class_2338.method_49637(this.position.x, this.position.y + 0.5, this.position.z));
        if (!in.method_26215()) {
            return in;
        }

        return this.level.method_8320(class_2338.method_49637(this.position.x, this.position.y - 0.5, this.position.z));
    }

    public Vector3f getRotation() {
        return this.rotation;
    }

    public float getRadius() {
        return this.radius;
    }

    public int getAge() {
        return this.age;
    }

    public int getLifetime() {
        return this.settings.particleLifetime();
    }

    public class_238 getBoundingBox() {
        return this.boundingBox;
    }

    public RenderData getRenderData() {
        return this.renderData;
    }

    public MolangEnvironment getEnvironment() {
        return this.environment.get();
    }

    public void vectorToRotation(double x, double y, double z) {
        this.rotation.set((float) Math.asin(y), (float) Math.atan2(x, z), 0);
    }

    public void setRadius(float radius) {
        this.radius = radius;
        this.updateBoundingBox();
    }

    public void setAge(int age) {
        this.age = age;
    }
}
