package foundry.veil.api.network;

import foundry.veil.api.network.handler.ClientPacketContext;
import foundry.veil.api.network.handler.PacketContext;
import foundry.veil.api.network.handler.ServerPacketContext;
import foundry.veil.impl.network.ClientPacketSink;
import net.minecraft.class_1297;
import net.minecraft.class_1923;
import net.minecraft.class_2338;
import net.minecraft.class_2586;
import net.minecraft.class_2596;
import net.minecraft.class_2602;
import net.minecraft.class_2658;
import net.minecraft.class_3215;
import net.minecraft.class_3218;
import net.minecraft.class_3222;
import net.minecraft.class_8042;
import net.minecraft.class_8710;
import net.minecraft.class_9129;
import net.minecraft.class_9139;
import net.minecraft.server.MinecraftServer;
import org.jetbrains.annotations.ApiStatus;
import org.jetbrains.annotations.Nullable;

import java.util.ArrayList;
import java.util.List;
import java.util.ServiceLoader;

/**
 * Manages packet registration and sending.
 *
 * @author Ocelot
 */
public interface VeilPacketManager {

    /**
     * The singleton instance of the {@link VeilPacketManager.Factory}. This is different on each loader.
     */
    VeilPacketManager.Factory FACTORY = ServiceLoader.load(VeilPacketManager.Factory.class).findFirst().orElseThrow(() -> new RuntimeException("Failed to load packet provider"));

    /**
     * Creates a {@link VeilPacketManager}.
     *
     * @param modId   The id of the mod creating the channel
     * @param version The NeoForge channel version
     * @return The packet manager
     */
    static VeilPacketManager create(String modId, String version) {
        return FACTORY.create(modId, version);
    }

    /**
     * Registers a new packet from the client to the server.
     *
     * @param id      The id of the packet
     * @param codec   The codec for encoding and decoding the packet
     * @param handler The handler method for the packet on the client
     * @param <T>     The type of packet to register
     */
    <T extends class_8710> void registerClientbound(class_8710.class_9154<T> id, class_9139<? super class_9129, T> codec, PacketHandler<ClientPacketContext, T> handler);

    /**
     * Registers a new packet from the server to the client.
     *
     * @param id      The id of the packet
     * @param codec   The codec for encoding and decoding the packet
     * @param handler The handler method for the packet on the server
     * @param <T>     The type of packet to register
     */
    <T extends class_8710> void registerServerbound(class_8710.class_9154<T> id, class_9139<? super class_9129, T> codec, PacketHandler<ServerPacketContext, T> handler);

    /**
     * @return A sink to send packets to the server
     */
    static PacketSink server() {
        return ClientPacketSink.INSTANCE;
    }

    /**
     * @return A sink to send packets to the specified player
     */
    static PacketSink player(class_3222 player) {
        return packet -> player.field_13987.method_14364(packet);
    }

    /**
     * Sends packets to all players in the specified level.
     *
     * @param level The level to send the packet to
     * @return A sink to send packets to all players in the dimension
     */
    static PacketSink level(class_3218 level) {
        return packet -> level.method_8503().method_3760().method_14589(packet, level.method_27983());
    }

    /**
     * Sends packets to all players in the area covered by the specified radius, around the specified coordinates, in the specified dimension, excluding the specified excluded player if present.
     *
     * @param excluded The player to exclude when sending the packet or <code>null</code> to send to all players
     * @param level    the level to send the packet around
     * @param x        the X position
     * @param y        the Y position
     * @param z        the Z position
     * @param radius   the maximum distance from the position in blocks
     * @return A sink to send packets to all players in the area
     */
    static PacketSink around(@Nullable class_3222 excluded, class_3218 level, double x, double y, double z, double radius) {
        return packet -> level.method_8503().method_3760().method_14605(excluded, x, y, z, radius, level.method_27983(), packet);
    }

    /**
     * Sends packets to all players in the server.
     *
     * @param server The server instance
     * @return A sink to send packets to all players
     */
    static PacketSink all(MinecraftServer server) {
        return packet -> server.method_3760().method_14581(packet);
    }

    /**
     * Sends packets to all players tracking the specified entity, excluding the entity.
     *
     * @param entity The entity to send tracking packets to
     * @return A sink to send packets to all players tracking the entity
     * @throws IllegalArgumentException if the entity is not in a server world
     */
    static PacketSink tracking(class_1297 entity) {
        return packet -> {
            if (!(entity.method_37908().method_8398() instanceof class_3215 chunkCache)) {
                throw new IllegalStateException("Cannot send clientbound payloads on the client");
            }

            chunkCache.method_18754(entity, packet);
        };
    }

    /**
     * Sends packets to all players tracking the specified entity, including the entity.
     *
     * @param entity The entity to send tracking packets to
     * @return A sink to send packets to all players tracking the entity and the entity
     * @throws IllegalArgumentException if the entity is not in a server world
     */
    static PacketSink trackingAndSelf(class_1297 entity) {
        return packet -> {
            if (!(entity.method_37908().method_8398() instanceof class_3215 chunkCache)) {
                throw new IllegalStateException("Cannot send clientbound payloads on the client");
            }

            chunkCache.method_18751(entity, packet);
        };
    }

    /**
     * Sends packets to all players tracking the chunk at the specified position in the specified level
     *
     * @param level The level to send the packet to
     * @param pos   The chunk to send to
     * @return A sink to send packets to all players tracking that chunk
     */
    static PacketSink tracking(class_3218 level, class_1923 pos) {
        return packet -> {
            for (class_3222 player : level.method_14178().field_17254.method_17210(pos, false)) {
                player.field_13987.method_14364(packet);
            }
        };
    }

    /**
     * Sends packets to all players tracking the chunk at the specified position in the specified level
     *
     * @param level The level to send the packet to
     * @param pos   The position to send to
     * @return A sink to send packets to all players tracking that chunk
     */
    static PacketSink tracking(class_3218 level, class_2338 pos) {
        return tracking(level, new class_1923(pos));
    }

    /**
     * Sends packets to all players tracking the chunks at the specified positions in the specified level
     *
     * @param level The level to send the packet to
     * @param min   The minimum position to send to
     * @param max   The maximum position to send to
     * @return A sink to send packets to all players tracking that chunk
     */
    static PacketSink tracking(class_3218 level, class_2338 min, class_2338 max) {
        return packet -> class_1923.method_19281(new class_1923(min), new class_1923(max))
                .flatMap(pos -> level.method_14178().field_17254.method_17210(pos, false).stream())
                .distinct()
                .forEach(player -> player.field_13987.method_14364(packet));
    }

    /**
     * Sends packets to all players tracking the specified block entity.
     *
     * @param blockEntity The block entity to send the packet to
     * @return A sink to send packets to all players tracking that block entity
     * @throws IllegalArgumentException if the block entity is not in a server level
     */
    static PacketSink tracking(class_2586 blockEntity) {
        if (!(blockEntity.method_10997() instanceof class_3218 serverLevel)) {
            throw new IllegalArgumentException("Only supported on server levels!");
        }
        return tracking(serverLevel, blockEntity.method_11016());
    }

    /**
     * Handles packets from the client/server.
     *
     * @param <T> The context to use
     * @param <P> The packet to handle
     */
    @FunctionalInterface
    interface PacketHandler<T extends PacketContext, P extends class_8710> {

        /**
         * Handles the specified packet.
         *
         * @param payload The packet payload
         * @param ctx     The sided context
         */
        void handlePacket(P payload, T ctx);
    }

    /**
     * Sends packets to players and automatically bundles payloads together.
     */
    @FunctionalInterface
    interface PacketSink {

        /**
         * Sends one or more payloads in a single packet.
         *
         * @param payloads All packets to send
         */
        default void sendPacket(class_8710... payloads) {
            if (payloads.length == 0) {
                return;
            }
            if (payloads.length == 1) {
                this.sendPacket(new class_2658(payloads[0]));
                return;
            }

            List<class_2596<? super class_2602>> packets = new ArrayList<>();
            for (class_8710 payload : payloads) {
                packets.add(new class_2658(payload));
            }
            this.sendPacket(new class_8042(packets));
        }

        /**
         * Sends a single packet.
         *
         * @param packet the packet to send
         */
        void sendPacket(class_2596<?> packet);
    }

    /**
     * Factory class for {@link VeilPacketManager registration providers}. <br>
     * This class is loaded using {@link ServiceLoader Service Loaders}, and only one
     * should exist per mod loader.
     */
    @ApiStatus.Internal
    interface Factory {

        /**
         * Creates a {@link VeilPacketManager}.
         *
         * @param modId   The id of the mod to register the channel under
         * @param version The NeoForge channel version
         * @return The packet manager
         */
        VeilPacketManager create(String modId, String version);
    }
}
