package foundry.veil.api.client.render.texture;

import com.mojang.logging.LogUtils;
import foundry.veil.api.client.render.VeilRenderSystem;
import org.jetbrains.annotations.NotNull;
import org.lwjgl.system.NativeResource;
import org.slf4j.Logger;

import javax.annotation.Nullable;
import net.minecraft.class_1011;
import net.minecraft.class_1060;
import net.minecraft.class_1084;
import net.minecraft.class_156;
import net.minecraft.class_2960;
import net.minecraft.class_3298;
import net.minecraft.class_3300;
import java.io.IOException;
import java.io.InputStream;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;

import static org.lwjgl.opengl.GL11C.GL_RGBA;

public class SimpleArrayTexture extends ArrayTexture implements VeilPreloadedTexture {

    private static final Logger LOGGER = LogUtils.getLogger();

    protected final class_2960[] locations;

    private CompletableFuture<TextureImage[]> imagesFuture;

    public SimpleArrayTexture(class_2960... locations) {
        if (locations.length == 0) {
            throw new IllegalStateException("SimpleArrayTexture requires at least one location");
        }
        this.locations = locations;
        this.imagesFuture = null;
    }

    @Override
    public CompletableFuture<?> preload(class_3300 resourceManager, Executor backgroundExecutor) {
        if (this.imagesFuture == null || this.imagesFuture.isDone()) {
            this.imagesFuture = CompletableFuture.supplyAsync(() -> TextureImage.load(resourceManager, this.locations), backgroundExecutor);
        }
        return this.imagesFuture;
    }

    @Override
    public void method_4625(@NotNull class_3300 resourceManager) throws IOException {
        TextureImage[] textureImages = this.getTextureImages(resourceManager);
        class_1011[] images = new class_1011[textureImages.length];
        try {
            for (int i = 0; i < textureImages.length; i++) {
                images[i] = textureImages[i].getImage();
            }
        } catch (IOException e) {
            for (TextureImage textureImage : textureImages) {
                textureImage.free();
            }
            throw e;
        }

        class_1084 texturemetadatasection = textureImages[0].getTextureMetadata();
        boolean blur;
        boolean clamp;
        if (texturemetadatasection != null) {
            blur = texturemetadatasection.method_4696();
            clamp = texturemetadatasection.method_4697();
        } else {
            blur = false;
            clamp = false;
        }

        VeilRenderSystem.renderThreadExecutor().execute(() -> {
            try {
                this.method_4527(blur, clamp);
                this.init(GL_RGBA, 0, images[0].method_4307(), images[0].method_4323(), images.length);
                this.upload(images);
            } finally {
                for (class_1011 image : images) {
                    image.close();
                }
            }
        });
    }

    @Override
    public void method_18169(@NotNull class_1060 textureManager, @NotNull class_3300 resourceManager, @NotNull class_2960 location, @NotNull Executor gameExecutor) {
        this.preload(resourceManager, class_156.method_18349()).thenRunAsync(() -> textureManager.method_4616(location, this), gameExecutor);
    }

    protected TextureImage[] getTextureImages(class_3300 resourceManager) {
        if (this.imagesFuture != null) {
            TextureImage[] images = this.imagesFuture.join();
            this.imagesFuture = null;
            return images;
        } else {
            return TextureImage.load(resourceManager, this.locations);
        }
    }

    public static class TextureImage implements NativeResource {

        private final class_1011 image;
        private final class_1084 metadata;
        private final IOException exception;

        public TextureImage(IOException exception) {
            this.image = null;
            this.metadata = null;
            this.exception = Objects.requireNonNull(exception, "exception");
        }

        public TextureImage(class_1011 image, @Nullable class_1084 metadata) {
            this.image = Objects.requireNonNull(image, "image");
            this.metadata = metadata;
            this.exception = null;
        }

        public static TextureImage[] load(class_3300 resourceManager, class_2960... locations) {
            TextureImage[] images = new TextureImage[locations.length];
            if (locations.length == 0) {
                return images;
            }

            int width = 0;
            int height = 0;
            for (int i = 0; i < locations.length; i++) {
                class_1011 image = null;
                try {
                    class_2960 location = locations[i];
                    class_3298 resource = resourceManager.getResourceOrThrow(location);

                    try (InputStream inputstream = resource.method_14482()) {
                        image = class_1011.method_4309(inputstream);
                    }
                    if (width == 0 || height == 0) {
                        width = image.method_4307();
                        height = image.method_4323();
                    } else if (image.method_4307() != width || image.method_4323() != height) {
                        throw new IOException("Layer " + i + " dimensions don't match");
                    }

                    class_1084 metadata = null;
                    try {
                        metadata = resource.method_14481().method_43041(class_1084.field_5344).orElse(null);
                    } catch (Exception e) {
                        LOGGER.warn("Failed reading metadata of: {}", location, e);
                    }

                    images[i] = new TextureImage(image, metadata);
                } catch (IOException e) {
                    if (image != null) {
                        image.close();
                    }
                    images[i] = new TextureImage(e);
                }
            }

            return images;
        }

        public class_1011 getImage() throws IOException {
            if (this.exception != null) {
                throw this.exception;
            } else {
                return this.image;
            }
        }

        public @Nullable class_1084 getTextureMetadata() {
            return this.metadata;
        }

        public @Nullable IOException getException() {
            return this.exception;
        }

        @Override
        public void free() {
            if (this.image != null) {
                this.image.close();
            }
        }

        public void throwIfError() throws IOException {
            if (this.exception != null) {
                throw this.exception;
            }
        }
    }
}
