package foundry.veil.api.client.render.post;

import com.google.common.collect.Iterables;
import com.google.gson.JsonElement;
import com.google.gson.JsonParser;
import com.google.gson.JsonSyntaxException;
import com.mojang.blaze3d.platform.GlStateManager;
import com.mojang.blaze3d.systems.RenderSystem;
import com.mojang.serialization.DataResult;
import com.mojang.serialization.JsonOps;
import foundry.veil.Veil;
import foundry.veil.VeilClient;
import foundry.veil.api.CodecReloadListener;
import foundry.veil.api.client.render.VeilRenderSystem;
import foundry.veil.api.client.render.VeilRenderer;
import foundry.veil.api.client.render.ext.VeilDebug;
import foundry.veil.api.client.render.framebuffer.AdvancedFbo;
import foundry.veil.api.client.render.framebuffer.FramebufferStack;
import foundry.veil.api.client.render.framebuffer.VeilFramebuffers;
import foundry.veil.api.client.render.post.stage.CompositePostPipeline;
import foundry.veil.api.client.render.shader.program.ShaderProgram;
import foundry.veil.api.event.VeilRenderLevelStageEvent;
import foundry.veil.impl.client.render.pipeline.PostPipelineContext;
import foundry.veil.platform.VeilClientPlatform;
import org.jetbrains.annotations.ApiStatus;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.lwjgl.system.NativeResource;

import java.io.IOException;
import java.io.Reader;
import java.util.*;
import net.minecraft.class_2960;
import net.minecraft.class_310;
import net.minecraft.class_3298;
import net.minecraft.class_3300;
import net.minecraft.class_3695;
import net.minecraft.class_7654;

import static org.lwjgl.opengl.GL11.GL_COLOR_BUFFER_BIT;
import static org.lwjgl.opengl.GL11.GL_NEAREST;
import static org.lwjgl.opengl.GL11C.GL_ALWAYS;
import static org.lwjgl.opengl.GL11C.GL_LEQUAL;

/**
 * <p>Manages all post pipelines.</p>
 * <p>Post Pipelines are a single "effect" that can be applied.
 * For example, a vanilla Minecraft creeper effect can be added using {@link #add(int, class_2960)}</p>
 *
 * @author Ocelot
 */
public class PostProcessingManager extends CodecReloadListener<CompositePostPipeline> implements NativeResource {

    private static final Comparator<ProfileEntry> PIPELINE_SORTER = Comparator.comparingInt(ProfileEntry::getPriority).reversed();
    private static final class_2960 POST = Veil.veilPath("post");

    private final PostPipelineContext context;
    private final List<ProfileEntry> activePipelines;
    private final List<ProfileEntry> activePipelinesView;
    private final Map<class_2960, CompositePostPipeline> pipelines;
    private int enabledBuffers;

    /**
     * Creates a new instance of the post-processing manager.
     */
    public PostProcessingManager() {
        super(CompositePostPipeline.CODEC, class_7654.method_45114("pinwheel/post"));
        this.context = new PostPipelineContext();
        this.activePipelines = new LinkedList<>();
        this.activePipelinesView = Collections.unmodifiableList(this.activePipelines);
        this.pipelines = new HashMap<>();
        this.enabledBuffers = 0;
    }

    /**
     * Checks to see if the specified pipeline is active.
     *
     * @param pipeline The pipeline to check for
     * @return Whether that pipeline is active
     */
    public boolean isActive(class_2960 pipeline) {
        return this.activePipelines.stream().anyMatch(entry -> entry.pipeline.equals(pipeline));
    }

    /**
     * Adds the specified pipeline under the default priority of <code>1000</code>.
     * A higher priority indicates the pipeline should be run earlier than lower priority pipelines.
     *
     * @param pipeline The pipeline to add
     * @return Whether the pipeline was added or had a priority change
     */
    public boolean add(class_2960 pipeline) {
        return this.add(1000, pipeline);
    }

    /**
     * Adds the specified pipeline with the specified priority.
     * A higher priority indicates the pipeline should be run earlier than lower priority pipelines.
     *
     * @param priority The priority to set the pipeline to. The default priority is <code>1000</code>
     * @param pipeline The pipeline to add
     * @return Whether the pipeline was added or had a priority change
     */
    public boolean add(int priority, class_2960 pipeline) {
        if (this.activePipelines.stream().anyMatch(entry -> entry.priority == priority && entry.pipeline.equals(pipeline))) {
            return false;
        }
        this.activePipelines.removeIf(entry -> entry.pipeline.equals(pipeline));
        this.activePipelines.add(new ProfileEntry(pipeline, priority));
        return true;
    }

    /**
     * Removes the specified pipeline from the active profiles.
     *
     * @param pipeline The pipeline to remove
     * @return If the pipeline was previously active
     */
    public boolean remove(class_2960 pipeline) {
        return this.activePipelines.removeIf(entry -> entry.pipeline.equals(pipeline));
    }

    /**
     * Retrieves a post pipeline by name.
     *
     * @param pipeline The pipeline to get
     * @return The pipeline found or <code>null</code> if it doesn't exist
     */
    public @Nullable PostPipeline getPipeline(class_2960 pipeline) {
        return this.pipelines.get(pipeline);
    }

    @ApiStatus.Internal
    public void endFrame() {
        // Disable any buffers that didn't draw this frame
        VeilRenderSystem.renderer().getDynamicBufferManger().setActiveBuffers(POST, this.enabledBuffers);
    }

    private void setup() {
        RenderSystem.enableDepthTest();
        RenderSystem.depthFunc(GL_ALWAYS);
        RenderSystem.depthMask(false);
        RenderSystem.enableBlend();
        RenderSystem.blendFuncSeparate(GlStateManager.class_4535.ONE, GlStateManager.class_4534.ZERO, GlStateManager.class_4535.ONE, GlStateManager.class_4534.ZERO);
        FramebufferStack.push(POST);
    }

    private void clear() {
        ShaderProgram.unbind();
        RenderSystem.depthFunc(GL_LEQUAL);
        RenderSystem.disableDepthTest();
        RenderSystem.depthMask(true);
        RenderSystem.disableBlend();
        RenderSystem.defaultBlendFunc();
        FramebufferStack.pop(POST);
    }

    private void clearPipeline() {
        RenderSystem.colorMask(true, true, true, true);
        RenderSystem.depthFunc(GL_ALWAYS);
        RenderSystem.depthMask(false);
    }

    @ApiStatus.Internal
    public void runDefaultPipeline(@Nullable VeilRenderLevelStageEvent.Stage stage) {
        if (this.activePipelines.isEmpty()) {
            return;
        }

        VeilDebug debug = VeilDebug.get();
        if (stage != null) {
            debug.pushDebugGroup("Veil Post Processing (" + stage.getName() + ")");
        } else {
            debug.pushDebugGroup("Veil Post Processing");
        }

        VeilRenderer renderer = VeilRenderSystem.renderer();
        AdvancedFbo postFramebuffer = renderer.getFramebufferManager().getFramebuffer(VeilFramebuffers.POST);
        VeilClientPlatform platform = VeilClient.clientPlatform();
        this.context.begin();
        this.setup();
        int activeTexture = GlStateManager._getActiveTexture();

        this.activePipelines.sort(PIPELINE_SORTER);
        for (ProfileEntry entry : this.activePipelines) {
            class_2960 id = entry.getPipeline();
            CompositePostPipeline pipeline = this.pipelines.get(id);
            if (pipeline != null) {
                this.enabledBuffers |= pipeline.getDynamicBuffersMask();
                // The buffer hasn't been enabled yet, so wait until next frame
                if ((renderer.getDynamicBufferManger().getActiveBuffers() & this.enabledBuffers) != this.enabledBuffers) {
                    continue;
                }

                // Only draw in the appropriate stage
                if (pipeline.getRenderStage() != stage) {
                    continue;
                }

                platform.preVeilPostProcessing(id, pipeline, this.context);
                try {
                    pipeline.apply(this.context);
                    this.clearPipeline();
                    // Resolve back to main for the next pipeline
                    if (postFramebuffer != null) {
                        postFramebuffer.resolveToRenderTarget(
                                class_310.method_1551().method_1522(),
                                GL_COLOR_BUFFER_BIT,
                                GL_NEAREST
                        );
                    }
                } catch (Exception e) {
                    Veil.LOGGER.error("Error running pipeline {}", id, e);
                }
                platform.postVeilPostProcessing(id, pipeline, this.context);
            }
        }

        RenderSystem.activeTexture(activeTexture);
        this.clear();
        this.context.end();

        debug.popDebugGroup();
    }

    /**
     * Applies only the specified pipeline.
     * Copies the post framebuffer to the main framebuffer after running the pipeline.
     *
     * @param pipeline The pipeline to run
     */
    public void runPipeline(PostPipeline pipeline) {
        this.runPipeline(pipeline, true);
    }

    /**
     * Applies only the specified pipeline.
     *
     * @param pipeline    The pipeline to run
     * @param resolvePost Whether to copy the post framebuffer to the main framebuffer after running the pipeline
     */
    public void runPipeline(PostPipeline pipeline, boolean resolvePost) {
        VeilRenderer renderer = VeilRenderSystem.renderer();
        if (pipeline instanceof CompositePostPipeline compositePostPipeline) {
            this.enabledBuffers |= compositePostPipeline.getDynamicBuffersMask();
            // The buffer hasn't been enabled yet, so wait until next frame
            if ((renderer.getDynamicBufferManger().getActiveBuffers() & this.enabledBuffers) != this.enabledBuffers) {
                return;
            }
        }

        this.context.begin();
        this.setup();
        int activeTexture = GlStateManager._getActiveTexture();

        try {
            pipeline.apply(this.context);
            this.clearPipeline();
        } catch (Exception e) {
            Veil.LOGGER.error("Error running pipeline {}", pipeline, e);
        }

        RenderSystem.activeTexture(activeTexture);
        this.clear();
        this.context.end();

        AdvancedFbo postFramebuffer = resolvePost ? renderer.getFramebufferManager().getFramebuffer(VeilFramebuffers.POST) : null;
        if (postFramebuffer != null) {
            postFramebuffer.resolveToRenderTarget(
                    class_310.method_1551().method_1522(),
                    GL_COLOR_BUFFER_BIT,
                    GL_NEAREST
            );
        }
    }

    private CompositePostPipeline loadPipeline(class_3298 resource) throws IOException {
        try (Reader reader = resource.method_43039()) {
            JsonElement element = JsonParser.parseReader(reader);
            DataResult<CompositePostPipeline> result = this.codec.parse(JsonOps.INSTANCE, element);

            if (result.error().isPresent()) {
                throw new JsonSyntaxException(result.error().get().message());
            }
            return result.result().orElseThrow();
        }
    }

    @Override
    protected @NotNull Map<class_2960, CompositePostPipeline> method_18789(@NotNull class_3300 resourceManager, @NotNull class_3695 profilerFiller) {
        Map<class_2960, CompositePostPipeline> data = new HashMap<>();

        Map<class_2960, List<class_3298>> resources = this.converter.method_45116(resourceManager);
        for (Map.Entry<class_2960, List<class_3298>> entry : resources.entrySet()) {
            class_2960 location = entry.getKey();
            class_2960 id = this.converter.method_45115(location);

            if (entry.getValue().size() == 1) {
                try {
                    data.put(id, this.loadPipeline(Iterables.getOnlyElement(entry.getValue())));
                } catch (Exception e) {
                    Veil.LOGGER.error("Couldn't parse data file {} from {}", id, location, e);
                }
                continue;
            }

            List<CompositePostPipeline> pipelines = new ArrayList<>(entry.getValue().size());
            for (class_3298 resource : entry.getValue()) {
                try {
                    pipelines.add(this.loadPipeline(resource));
                } catch (Exception e) {
                    Veil.LOGGER.error("Couldn't parse data file {} from {}", id, location, e);
                }
            }

            // No pipelines loaded, so continue
            if (pipelines.isEmpty()) {
                continue;
            }

            // Only 1 pipeline successfully loaded
            if (pipelines.size() == 1) {
                data.put(id, Iterables.getOnlyElement(pipelines));
                continue;
            }

            int dynamicBuffers = 0;
            pipelines.sort(Comparator.comparingInt(CompositePostPipeline::getPriority));
            for (int i = 0; i < pipelines.size(); i++) {
                CompositePostPipeline pipeline = pipelines.get(i);
                dynamicBuffers |= pipeline.getDynamicBuffersMask();
                if (pipeline.isReplace()) {
                    pipelines = pipelines.subList(0, i + 1);
                    break;
                }
            }
            data.put(id, new CompositePostPipeline(pipelines.toArray(CompositePostPipeline[]::new), Collections.emptyMap(), Collections.emptyMap(), pipelines.getFirst().getRenderStage(), dynamicBuffers));
        }

        return data;
    }

    @Override
    protected void apply(@NotNull Map<class_2960, CompositePostPipeline> data, @NotNull class_3300 resourceManager, @NotNull class_3695 profilerFiller) {
        this.free();
        this.pipelines.putAll(data);
        Veil.LOGGER.info("Loaded {} post pipelines", this.pipelines.size());
    }

    @Override
    public void free() {
        this.pipelines.values().forEach(PostPipeline::free);
        this.pipelines.clear();
    }

    /**
     * @return The default context for post-processing
     */
    public PostPipeline.Context getPostPipelineContext() {
        return this.context;
    }

    /**
     * @return All available pipelines
     */
    public @NotNull Set<class_2960> getPipelines() {
        return this.pipelines.keySet();
    }

    /**
     * @return An immutable view of all active profiles and their priorities
     */
    public List<ProfileEntry> getActivePipelines() {
        return this.activePipelinesView;
    }

    /**
     * A single active profile.
     */
    public static class ProfileEntry {

        private final class_2960 pipeline;
        private int priority;

        public ProfileEntry(class_2960 pipeline, int priority) {
            this.pipeline = pipeline;
            this.priority = priority;
        }

        /**
         * @return The id of the pipeline shader
         */
        public class_2960 getPipeline() {
            return this.pipeline;
        }

        /**
         * @return The priority the profile is inserted at
         */
        public int getPriority() {
            return this.priority;
        }

        /**
         * Sets the priority this effect is applied at.
         *
         * @param priority The new priority
         */
        public void setPriority(int priority) {
            this.priority = priority;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            ProfileEntry that = (ProfileEntry) o;
            return this.priority == that.priority && Objects.equals(this.pipeline, that.pipeline);
        }

        @Override
        public int hashCode() {
            int result = this.pipeline.hashCode();
            result = 31 * result + this.priority;
            return result;
        }
    }
}
