package foundry.veil.api.client.registry;

import foundry.veil.ext.CompositeStateExtension;
import foundry.veil.mixin.accessor.RenderStateShardAccessor;
import org.jetbrains.annotations.ApiStatus;

import java.util.*;
import java.util.function.Predicate;
import net.minecraft.class_1921;
import net.minecraft.class_4668;

/**
 * <p>This allows custom render type stages to be registered. This allows custom code to be run for the setup and clear state of any render type.
 * {@link RenderTypeStageRegistry#addGenericStage(Predicate, class_4668...)} also allows arbitrary injection into any render type created.</p>
 * <p><strong>This should be called during mod construction/init.</strong></p>
 *
 * @author Ocelot
 */
public final class RenderTypeStageRegistry {

    private static final Map<String, List<class_4668>> STAGES = new HashMap<>();
    private static final Set<GenericStage> GENERIC_STAGES = new HashSet<>();
    private static final Set<class_1921.class_4687> CREATED_RENDER_TYPES = new HashSet<>();

    private RenderTypeStageRegistry() {
    }

    /**
     * Registers a render stage. The specified shards will be added to the specified render type during construction.
     *
     * @param renderType The render type to add the stage to
     * @param shards     The shards to add to all matching render types
     */
    public static synchronized void addStage(class_1921 renderType, class_4668... shards) {
        addStage(((RenderStateShardAccessor) renderType).getName(), shards);
    }

    /**
     * Registers a render stage. The specified shards will be added to the specified render type during construction.
     *
     * @param name   The name of the render type to add the stage to
     * @param shards The shards to add to all matching render types
     */
    public static synchronized void addStage(String name, class_4668... shards) {
        if (shards.length == 0) {
            throw new IllegalArgumentException("No shards provided");
        }
        STAGES.computeIfAbsent(name, unused -> new ArrayList<>()).addAll(Arrays.asList(shards));

        for (class_1921.class_4687 renderType : CREATED_RENDER_TYPES) {
            inject(renderType);
        }
    }

    /**
     * Registers a render stage. The specified shards will be added to all render types that match the specified filter during construction.
     *
     * @param filter The filter for what render types to add the stage to
     * @param shards The shards to add to all matching render types
     */
    public static synchronized void addGenericStage(Predicate<class_1921.class_4687> filter, class_4668... shards) {
        if (shards.length == 0) {
            throw new IllegalArgumentException("No shards provided");
        }
        GENERIC_STAGES.add(new GenericStage(filter, shards));

        for (class_1921.class_4687 renderType : CREATED_RENDER_TYPES) {
            inject(renderType);
        }
    }

    @SuppressWarnings({"UnreachableCode", "DataFlowIssue"})
    @ApiStatus.Internal
    public static void inject(class_1921.class_4687 renderType) {
        List<class_4668> shards = STAGES.get(((RenderStateShardAccessor) (Object) renderType).getName());
        if (shards != null) {
            shards = new ArrayList<>(shards);
        }
        for (GenericStage stage : GENERIC_STAGES) {
            if (stage.filter.test(renderType)) {
                if (shards == null) {
                    shards = new ArrayList<>(Arrays.asList(stage.shards));
                    continue;
                }

                shards.addAll(Arrays.asList(stage.shards));
            }
        }

        if (shards != null) {
            ((CompositeStateExtension) (Object) renderType.method_35784()).veil$addShards(shards);
        }
        CREATED_RENDER_TYPES.add(renderType);
    }

    private record GenericStage(Predicate<class_1921.class_4687> filter, class_4668... shards) {
    }
}
