package foundry.veil.api.client.ui.util;

import com.mojang.blaze3d.systems.RenderSystem;
import java.util.ArrayDeque;
import java.util.Deque;
import net.minecraft.class_310;
import net.minecraft.class_3532;

/**
 * A utility class to manage scissor clipping regions.
 * This allows for restricting rendering to specific rectangular areas.
 */
public class ScissorStack {
    private final Deque<ScissorRegion> regions = new ArrayDeque<>();

    public ScissorStack() {
    }

    /**
     * Pushes a new scissor clipping region onto the stack.
     * The region is automatically constrained by any existing regions on the stack.
     *
     * @param x The x-coordinate of the top-left corner of the region.
     * @param y The y-coordinate of the top-left corner of the region.
     * @param width  The width of the region.
     * @param height The height of the region.
     */
    public void push(int x, int y, int width, int height) {
        if (!regions.isEmpty()) {
            ScissorRegion parent = regions.peek();
            int x2 = x + width;
            x = class_3532.method_15340(x, parent.x, parent.x + parent.width);
            width = class_3532.method_15340(x2, parent.x, parent.x + parent.width) - x;
            int y2 = y + height;
            y = class_3532.method_15340(y, parent.y, parent.y + parent.height);
            height = class_3532.method_15340(y2, parent.y, parent.y + parent.height) - y;
        }

        ScissorRegion region = new ScissorRegion(x, y, width, height);
        regions.push(region);
        region.apply();
    }

    /**
     * Removes the top scissor clipping region from the stack.
     * If there are any regions remaining, the previous region is reapplied.
     */
    public void pop() {
        regions.pop();
        RenderSystem.disableScissor();
        if (!regions.isEmpty()) {
            regions.peek().apply();
        }
    }

    /**
     * Represents a single scissor clipping region.
     */
    private static class ScissorRegion {
        int x, y, width, height;

        public ScissorRegion(int x, int y, int width, int height) {
            this.x = x;
            this.y = y;
            this.width = width;
            this.height = height;
        }

        /**
         * Applies this scissor region to the rendering system.
         */
        void apply() {
            double scale = class_310.method_1551().method_22683().method_4495();
            int screenY = (int) ((class_310.method_1551().method_22683().method_4506() - (y + height)) * scale);
            RenderSystem.enableScissor((int) (x * scale), screenY, (int) (width * scale), (int) (height * scale));
        }
    }
}