package foundry.veil.api.client.editor;

import foundry.veil.Veil;
import foundry.veil.api.client.registry.VeilResourceEditorRegistry;
import foundry.veil.api.resource.editor.ResourceFileEditor;
import foundry.veil.api.util.CompositeReloadListener;
import imgui.ImFont;
import imgui.ImGui;
import imgui.flag.ImGuiCol;
import imgui.type.ImBoolean;
import org.jetbrains.annotations.ApiStatus;
import org.jetbrains.annotations.NotNull;

import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import net.minecraft.class_2561;
import net.minecraft.class_2960;
import net.minecraft.class_3300;
import net.minecraft.class_3302;
import net.minecraft.class_3304;
import net.minecraft.class_3695;

/**
 * <p>Manages all editors for Veil. Editors are ImGui powered panels that can be dynamically registered and unregistered with {@link #add(Editor)}.</p>
 *
 * @author Ocelot
 */
public class EditorManager implements class_3302 {

    public static final class_2960 DEFAULT_FONT = Veil.veilPath("jetbrains_mono");

    private final Map<Editor, ImBoolean> editors;
    private final EditorFontManager fonts;
    private boolean enabled;

    @ApiStatus.Internal
    public EditorManager(class_3304 resourceManager) {
        this.editors = new TreeMap<>(Comparator.comparing(editor -> editor.getClass().getSimpleName()));
        this.fonts = new EditorFontManager();
        this.enabled = false;

        resourceManager.method_14477(this);
    }

    public ImFont getFont(class_2960 name, boolean bold, boolean italic) {
        return this.fonts.getFont(name, bold, italic);
    }

    public ImFont getFont(boolean bold, boolean italic) {
        return this.getFont(DEFAULT_FONT, bold, italic);
    }

    @ApiStatus.Internal
    public void render() {
        if (!this.enabled) {
            return;
        }

        if (ImGui.beginMainMenuBar()) {
            ImFont font = ImGui.getFont();
            float dingleWidth = font.calcTextSizeAX(ImGui.getFontSize(), Float.MAX_VALUE, 0, " Veil ") + 4;
            float dingleHeight = ImGui.getTextLineHeightWithSpacing() + 2;
            ImGui.getWindowDrawList().addRectFilled(0f, 0f, dingleWidth, dingleHeight, ImGui.getColorU32(ImGuiCol.FrameBgHovered));
            ImGui.text("Veil ");

            for (Map.Entry<Editor, ImBoolean> entry : this.editors.entrySet()) {
                Editor editor = entry.getKey();
                class_2561 group = editor.getGroup();
                if (group == null) {
                    if (Veil.platform().isDevelopmentEnvironment()) {
                        Veil.LOGGER.error("Editor '{}' should return Editor#DEFAULT_GROUP instead of null", editor.getClass());
                    }
                    group = Editor.DEFAULT_GROUP;
                }
                if (ImGui.beginMenu(group.getString())) {
                    ImBoolean enabled = entry.getValue();

                    ImGui.beginDisabled(!editor.isEnabled());
                    if (ImGui.menuItem(editor.getDisplayName().getString(), null, enabled.get())) {
                        if (!enabled.get()) {
                            this.show(editor);
                        } else {
                            this.hide(editor);
                        }
                    }
                    ImGui.endDisabled();
                    ImGui.endMenu();
                }
            }

            for (Map.Entry<Editor, ImBoolean> entry : this.editors.entrySet()) {
                Editor editor = entry.getKey();
                if (entry.getValue().get() && editor.isMenuBarEnabled()) {
                    ImGui.separator();
                    ImGui.textColored(0xFFAAAAAA, editor.getDisplayName().getString());
                    editor.renderMenuBar();
                }
            }

            ImGui.endMainMenuBar();
        }

        for (Map.Entry<Editor, ImBoolean> entry : this.editors.entrySet()) {
            Editor editor = entry.getKey();
            ImBoolean enabled = entry.getValue();

            if (!editor.isEnabled()) {
                enabled.set(false);
            }
            if (!enabled.get()) {
                continue;
            }

            editor.render();
        }

        for (ResourceFileEditor<?> editor : VeilResourceEditorRegistry.REGISTRY) {
            editor.render();
        }
    }

    @ApiStatus.Internal
    public void renderLast() {
        if (!this.enabled) {
            return;
        }

        for (Map.Entry<Editor, ImBoolean> entry : this.editors.entrySet()) {
            Editor editor = entry.getKey();
            ImBoolean enabled = entry.getValue();
            if (enabled.get()) {
                editor.renderLast();
            }
        }
    }

    public void show(Editor editor) {
        ImBoolean enabled = this.editors.get(editor);
        if (enabled != null && !enabled.get()) {
            editor.onShow();
            enabled.set(true);
        }
    }

    public void hide(Editor editor) {
        ImBoolean enabled = this.editors.get(editor);
        if (enabled != null && enabled.get()) {
            editor.onHide();
            enabled.set(false);
        }
    }

    public boolean isVisible(Editor editor) {
        ImBoolean visible = this.editors.get(editor);
        return visible != null && visible.get();
    }

    public synchronized void add(Editor editor) {
        this.editors.computeIfAbsent(editor, unused -> new ImBoolean());
    }

    public synchronized void remove(Editor editor) {
        this.hide(editor);
        this.editors.remove(editor);
    }

    /**
     * Toggles visibility of the ImGui overlay.
     */
    public void toggle() {
        this.enabled = !this.enabled;
    }

    /**
     * @return Whether the overlay is active
     */
    public boolean isEnabled() {
        return this.enabled;
    }

    /**
     * Sets whether the overlay should be active.
     *
     * @param enabled Whether to enable the ImGui overlay
     */
    public void setEnabled(boolean enabled) {
        this.enabled = enabled;
    }

    @Override
    public @NotNull CompletableFuture<Void> method_25931(@NotNull class_4045 preparationBarrier, @NotNull class_3300 resourceManager, @NotNull class_3695 prepareProfiler, @NotNull class_3695 applyProfiler, @NotNull Executor backgroundExecutor, @NotNull Executor gameExecutor) {
        List<class_3302> listeners = new ArrayList<>(this.editors.size());
        listeners.add(this.fonts);
        for (Editor editor : this.editors.keySet()) {
            if (editor instanceof class_3302 listener) {
                listeners.add(listener);
            }
        }
        for (ResourceFileEditor<?> editor : VeilResourceEditorRegistry.REGISTRY) {
            if (editor instanceof class_3302 listener) {
                listeners.add(listener);
            }
        }
        class_3302 listener = CompositeReloadListener.of(listeners.toArray(class_3302[]::new));
        return listener.method_25931(preparationBarrier, resourceManager, prepareProfiler, applyProfiler, backgroundExecutor, gameExecutor);
    }
}
