package foundry.veil.api.resource;

import com.mojang.datafixers.util.Pair;
import com.mojang.serialization.Lifecycle;
import foundry.veil.mixin.accessor.RegistryDataAccessor;
import org.jetbrains.annotations.ApiStatus;
import org.jetbrains.annotations.Nullable;

import java.io.PrintWriter;
import java.io.StringWriter;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import net.minecraft.class_2378;
import net.minecraft.class_2385;
import net.minecraft.class_2960;
import net.minecraft.class_3300;
import net.minecraft.class_5321;
import net.minecraft.class_5455;
import net.minecraft.class_6903;
import net.minecraft.class_7655;

/**
 * Allows the creation of normal vanilla dynamic registries. They can be created client or server side depending on the use case.
 *
 * @author Ocelot
 */
public class VeilDynamicRegistry {

    private static final ThreadLocal<Boolean> LOADING = ThreadLocal.withInitial(() -> false);

    /**
     * Loads the specified registries from the normal registry folder.
     *
     * @param resourceManager The manager for all resources
     * @param registries      The registries to load
     * @return All loaded registries and their errors
     */
    public static Data loadRegistries(class_3300 resourceManager, Collection<class_7655.class_7657<?>> registries) {
        Map<class_5321<?>, Exception> errors = new HashMap<>();

        LOADING.set(true);
        List<Pair<class_2385<?>, class_7655.class_7656>> loaders = registries.stream().map(data -> ((RegistryDataAccessor) (Object) data).invokeCreate(Lifecycle.stable(), errors)).toList();
        class_6903.class_7863 lookup = class_7655.method_46619(class_5455.field_40585, loaders);
        loaders.forEach(pair -> pair.getSecond().load(resourceManager, lookup));
        loaders.forEach(pair -> {
            class_2378<?> registry = pair.getFirst();

            try {
                registry.method_40276();
            } catch (Exception e) {
                errors.put(registry.method_30517(), e);
            }
        });
        LOADING.set(false);

        class_5455.class_6890 registryAccess = new class_5455.class_6891(loaders.stream().map(Pair::getFirst).toList()).method_40316();
        return new Data(registryAccess, errors);
    }

    /**
     * Prints all errors from loading registries into a string.
     *
     * @param errors The errors to print
     * @return The errors or <code>null</code> if there were no errors
     */
    public static @Nullable String printErrors(Map<class_5321<?>, Exception> errors) {
        if (errors.isEmpty()) {
            return null;
        }

        StringWriter stringWriter = new StringWriter();
        PrintWriter printWriter = new PrintWriter(stringWriter);
        Map<class_2960, Map<class_2960, Exception>> sortedErrors = errors.entrySet().stream().collect(Collectors.groupingBy(entry -> entry.getKey().method_41185(), Collectors.toMap(entry -> entry.getKey().method_29177(), Map.Entry::getValue)));
        sortedErrors.entrySet().stream().sorted(Map.Entry.comparingByKey()).forEach(registryError -> {
            printWriter.printf("%n> %d Errors in registry %s:", registryError.getValue().size(), registryError.getKey());
            registryError.getValue().entrySet().stream().sorted(Map.Entry.comparingByKey()).forEach(elementError -> {
                Throwable error = elementError.getValue();
                while (error.getCause() != null) {
                    error = error.getCause();
                }
                printWriter.printf("%n>> Error in element %s: %s", elementError.getKey(), error.getMessage());
            });
        });
        printWriter.flush();
        return stringWriter.toString();
    }

    @ApiStatus.Internal
    public static boolean isLoading() {
        return LOADING.get();
    }

    public record Data(class_5455 registryAccess, Map<class_5321<?>, Exception> errors) {
    }
}
