package foundry.veil.api.client.render.shader;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.JsonParseException;
import com.google.gson.JsonSyntaxException;
import foundry.veil.Veil;
import foundry.veil.api.client.render.VeilRenderSystem;
import foundry.veil.api.client.render.VeilRenderer;
import foundry.veil.api.client.render.framebuffer.FramebufferManager;
import foundry.veil.api.client.render.post.PostProcessingManager;
import foundry.veil.api.client.render.shader.definition.ShaderPreDefinitions;
import foundry.veil.api.client.render.shader.processor.ShaderCPreprocessor;
import foundry.veil.api.client.render.shader.processor.ShaderCustomProcessor;
import foundry.veil.api.client.render.shader.processor.ShaderModifyProcessor;
import foundry.veil.api.client.render.shader.program.ProgramDefinition;
import foundry.veil.api.client.render.shader.program.ShaderProgram;
import it.unimi.dsi.fastutil.ints.Int2ObjectMap;
import org.jetbrains.annotations.Nullable;

import java.io.*;
import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import net.minecraft.class_156;
import net.minecraft.class_2960;
import net.minecraft.class_310;
import net.minecraft.class_3298;
import net.minecraft.class_3300;
import net.minecraft.class_3302;
import net.minecraft.class_3518;
import net.minecraft.class_3694;
import net.minecraft.class_3695;
import net.minecraft.class_5912;
import net.minecraft.class_7654;

import static org.lwjgl.opengl.GL20C.GL_FRAGMENT_SHADER;
import static org.lwjgl.opengl.GL20C.GL_VERTEX_SHADER;
import static org.lwjgl.opengl.GL32C.GL_GEOMETRY_SHADER;
import static org.lwjgl.opengl.GL40C.GL_TESS_CONTROL_SHADER;
import static org.lwjgl.opengl.GL40C.GL_TESS_EVALUATION_SHADER;
import static org.lwjgl.opengl.GL43C.GL_COMPUTE_SHADER;

/**
 * <p>Manages all shaders and compiles them automatically.</p>
 * <p>Shaders can be recompiled using {@link #recompile(class_2960, class_5912)} or
 * {@link #recompile(class_2960, class_5912, ShaderCompiler)} to use a custom compiler.</p>
 *
 * @author Ocelot
 * @see ShaderCompiler
 */
public class ShaderManager implements class_3302, Closeable {

    private static final Gson GSON = new GsonBuilder()
            .registerTypeAdapter(class_2960.class, new class_2960.class_2961())
            .registerTypeAdapter(ProgramDefinition.class, new ProgramDefinition.Deserializer())
            .create();

    public static final class_7654 INCLUDE_LISTER = new class_7654("pinwheel/shaders/include", ".glsl");
    public static final ShaderSourceSet PROGRAM_SET = new ShaderSourceSet("pinwheel/shaders/program");
    public static final ShaderSourceSet DEFERRED_SET = new ShaderSourceSet("pinwheel/shaders/deferred");

    private static final Map<Integer, String> TYPES = Map.of(
            GL_VERTEX_SHADER, "vertex",
            GL_TESS_CONTROL_SHADER, "tesselation_control",
            GL_TESS_EVALUATION_SHADER, "tesselation_evaluation",
            GL_GEOMETRY_SHADER, "geometry",
            GL_FRAGMENT_SHADER, "fragment",
            GL_COMPUTE_SHADER, "compute"
    );

    private final ShaderSourceSet sourceSet;
    private final ShaderPreDefinitions definitions;
    private final Map<class_2960, ShaderProgram> shaders;
    private final Map<class_2960, ShaderProgram> shadersView;
    private final Set<class_2960> dirtyShaders;
    private CompletableFuture<Void> reloadFuture;
    private CompletableFuture<Void> recompileFuture;

    /**
     * Creates a new shader manager.
     *
     * @param sourceSet            The source set to load all shaders from
     * @param shaderPreDefinitions The set of shader pre-definitions
     */
    public ShaderManager(ShaderSourceSet sourceSet, ShaderPreDefinitions shaderPreDefinitions) {
        this.sourceSet = sourceSet;
        this.definitions = shaderPreDefinitions;
        this.definitions.addListener(this::onDefinitionChanged);
        this.shaders = new HashMap<>();
        this.shadersView = Collections.unmodifiableMap(this.shaders);
        this.dirtyShaders = new HashSet<>();
        this.reloadFuture = CompletableFuture.completedFuture(null);
        this.recompileFuture = CompletableFuture.completedFuture(null);
    }

    private void onDefinitionChanged(String definition) {
        this.shaders.values().forEach(shader -> {
            if (shader.getDefinitionDependencies().contains(definition)) {
                Veil.LOGGER.debug("{} changed, recompiling {}", definition, shader.getId());
                this.scheduleRecompile(shader.getId());
            }
        });
    }

    private ProgramDefinition parseDefinition(class_2960 id, class_5912 provider) throws IOException {
        try (Reader reader = provider.openAsReader(this.sourceSet.getShaderDefinitionLister().method_45112(id))) {
            ProgramDefinition definition = class_3518.method_15276(GSON, reader, ProgramDefinition.class);
            if (definition.vertex() == null &&
                    definition.tesselationControl() == null &&
                    definition.tesselationEvaluation() == null &&
                    definition.geometry() == null &&
                    definition.fragment() == null &&
                    definition.compute() == null) {
                throw new JsonSyntaxException("Shader programs must define at least 1 shader type");
            }

            return definition;
        } catch (JsonParseException e) {
            throw new IOException(e);
        }
    }

    private void readShader(class_3300 resourceManager, Map<class_2960, ProgramDefinition> definitions, Map<class_2960, class_3298> shaderSources, class_2960 id) {
        Set<class_2960> checkedSources = new HashSet<>();

        try {
            ProgramDefinition definition = this.parseDefinition(id, resourceManager);
            if (definitions.put(id, definition) != null) {
                throw new IllegalStateException("Duplicate shader ignored with ID " + id);
            }

            for (Int2ObjectMap.Entry<ProgramDefinition.ShaderSource> shader : definition.shaders().int2ObjectEntrySet()) {
                class_7654 typeConverter = this.sourceSet.getTypeConverter(shader.getIntKey());
                class_2960 location = typeConverter.method_45112(shader.getValue().location());

                if (!checkedSources.add(location)) {
                    continue;
                }

                class_3298 resource = resourceManager.getResourceOrThrow(location);
                try (InputStream stream = resource.method_14482()) {
                    byte[] source = stream.readAllBytes();
                    class_3298 fileResource = new class_3298(resource.method_45304(), () -> new ByteArrayInputStream(source));
                    shaderSources.put(location, fileResource);
                } catch (Throwable t) {
                    throw new IOException("Failed to load " + getTypeName(shader.getIntKey()) + " shader", t);
                }
            }
        } catch (IOException | IllegalArgumentException | JsonParseException e) {
            Veil.LOGGER.error("Couldn't parse shader {} from {}", id, this.sourceSet.getShaderDefinitionLister().method_45112(id), e);
        }
    }

    private Map<class_2960, class_3298> readIncludes(class_3300 resourceManager) {
        Map<class_2960, class_3298> shaderSources = new HashMap<>();
        Set<class_2960> checkedSources = new HashSet<>();

        for (Map.Entry<class_2960, class_3298> entry : INCLUDE_LISTER.method_45113(resourceManager).entrySet()) {
            class_2960 location = entry.getKey();
            class_2960 id = INCLUDE_LISTER.method_45115(location);

            if (!checkedSources.add(location)) {
                continue;
            }

            try {
                class_3298 resource = resourceManager.getResourceOrThrow(location);
                try (InputStream stream = resource.method_14482()) {
                    byte[] source = stream.readAllBytes();
                    class_3298 fileResource = new class_3298(resource.method_45304(), () -> new ByteArrayInputStream(source));
                    shaderSources.put(location, fileResource);
                }
            } catch (IOException | IllegalArgumentException | JsonParseException e) {
                Veil.LOGGER.error("Couldn't parse shader import {} from {}", id, location, e);
            }
        }

        return shaderSources;
    }

    private void compile(ShaderProgram program, ProgramDefinition definition, ShaderCompiler compiler) {
        class_2960 id = program.getId();
        try {
            program.compile(new ShaderCompiler.Context(this.definitions, this.sourceSet, definition), compiler);
        } catch (ShaderException e) {
            Veil.LOGGER.error("Failed to create shader {}: {}", id, e.getMessage());
            String error = e.getGlError();
            if (error != null) {
                Veil.LOGGER.warn(error);
            }
        } catch (Exception e) {
            Veil.LOGGER.error("Failed to create shader: {}", id, e);
        }
    }

    private ShaderCompiler addProcessors(ShaderCompiler compiler, class_5912 provider) {
        return compiler.addDefaultProcessors()
                // We need imports to keep their macros and pre-processor directives before the full shader is created
                .addPreprocessor(new ShaderCPreprocessor(), false)
                .addPreprocessor(new ShaderModifyProcessor(), false)
                .addPreprocessor(new ShaderCustomProcessor(provider), false)
                .addPreprocessor(new ShaderCPreprocessor(), false);
    }

    /**
     * Attempts to recompile the shader with the specified id.
     *
     * @param id       The id of the shader to recompile
     * @param provider The source of resources
     */
    public void recompile(class_2960 id, class_5912 provider) {
        try (ShaderCompiler compiler = this.addProcessors(ShaderCompiler.direct(provider), provider)) {
            this.recompile(id, provider, compiler);
        }
    }

    /**
     * Attempts to recompile the shader with the specified id.
     *
     * @param id       The id of the shader to recompile
     * @param provider The source of resources
     * @param compiler The compiler instance to use. If unsure, use {@link #recompile(class_2960, class_5912)}
     */
    public void recompile(class_2960 id, class_5912 provider, ShaderCompiler compiler) {
        ShaderProgram program = this.shaders.get(id);
        if (program == null) {
            Veil.LOGGER.error("Failed to recompile unknown shader: {}", id);
            return;
        }

        try {
            this.compile(program, this.parseDefinition(id, provider), compiler);
        } catch (Exception e) {
            Veil.LOGGER.error("Failed to read shader definition: {}", id, e);
        }
    }

    /**
     * Sets a global shader value.
     *
     * @param setter The setter for shaders
     */
    public void setGlobal(Consumer<ShaderProgram> setter) {
        this.shaders.values().forEach(setter);
    }

    /**
     * Retrieves a shader by the specified id.
     *
     * @param id The id of the shader to retrieve
     * @return The retrieved shader or <code>null</code> if there is no valid shader with that id
     */
    public @Nullable ShaderProgram getShader(class_2960 id) {
        return this.shaders.get(id);
    }

    /**
     * @return All shader programs registered
     */
    public Map<class_2960, ShaderProgram> getShaders() {
        return this.shadersView;
    }

    /**
     * @return The source set all shaders are loaded from
     */
    public ShaderSourceSet getSourceSet() {
        return this.sourceSet;
    }

    private ReloadState prepare(class_3300 resourceManager, Collection<class_2960> shaders) {
        Map<class_2960, ProgramDefinition> definitions = new HashMap<>();
        Map<class_2960, class_3298> shaderSources = new HashMap<>();

        for (class_2960 key : shaders) {
            this.readShader(resourceManager, definitions, shaderSources, key);
        }
        shaderSources.putAll(this.readIncludes(resourceManager));

        return new ReloadState(definitions, shaderSources);
    }

    private void apply(ShaderManager.ReloadState reloadState) {
        this.shaders.values().forEach(ShaderProgram::free);
        this.shaders.clear();

        class_5912 sourceProvider = loc -> Optional.ofNullable(reloadState.shaderSources().get(loc));
        try (ShaderCompiler compiler = this.addProcessors(ShaderCompiler.cached(sourceProvider), sourceProvider)) {
            for (Map.Entry<class_2960, ProgramDefinition> entry : reloadState.definitions().entrySet()) {
                class_2960 id = entry.getKey();
                ShaderProgram program = ShaderProgram.create(id);
                this.compile(program, entry.getValue(), compiler);
                this.shaders.put(id, program);
            }
        }

        VeilRenderSystem.finalizeShaderCompilation();

        Veil.LOGGER.info("Loaded {} shaders from: {}", this.shaders.size(), this.sourceSet.getFolder());
    }

    private void applyRecompile(ShaderManager.ReloadState reloadState, Collection<class_2960> shaders) {
        class_5912 sourceProvider = loc -> Optional.ofNullable(reloadState.shaderSources().get(loc));
        try (ShaderCompiler compiler = this.addProcessors(ShaderCompiler.cached(sourceProvider), sourceProvider)) {
            for (Map.Entry<class_2960, ProgramDefinition> entry : reloadState.definitions().entrySet()) {
                class_2960 id = entry.getKey();
                ShaderProgram program = this.getShader(id);
                if (program == null) {
                    Veil.LOGGER.warn("Failed to recompile shader: {}", id);
                    continue;
                }
                this.compile(program, entry.getValue(), compiler);
            }
        }

        VeilRenderSystem.finalizeShaderCompilation();

        Veil.LOGGER.info("Recompiled {} shaders from: {}", shaders.size(), this.sourceSet.getFolder());
    }

    private void scheduleRecompile(int attempt) {
        class_310 client = class_310.method_1551();
        client.method_18858(() -> {
            if (!this.recompileFuture.isDone()) {
                return;
            }

            Set<class_2960> shaders;
            synchronized (this.dirtyShaders) {
                shaders = new HashSet<>(this.dirtyShaders);
                this.dirtyShaders.clear();
            }
            this.recompileFuture = CompletableFuture.supplyAsync(() -> this.prepare(client.method_1478(), shaders), class_156.method_18349())
                    .thenAcceptAsync(state -> this.applyRecompile(state, shaders), client)
                    .handle((value, e) -> {
                        if (e != null) {
                            Veil.LOGGER.error("Error recompiling shaders", e);
                        }

                        synchronized (this.dirtyShaders) {
                            if (this.dirtyShaders.isEmpty()) {
                                return value;
                            }
                        }

                        if (attempt >= 3) {
                            Veil.LOGGER.error("Failed to recompile shaders after {} attempts", attempt);
                            return value;
                        }

                        this.scheduleRecompile(attempt + 1);
                        return value;
                    });
        });
    }

    /**
     * Schedules a shader recompilation on the next loop iteration.
     *
     * @param shader The shader to recompile
     */
    public void scheduleRecompile(class_2960 shader) {
        synchronized (this.dirtyShaders) {
            this.dirtyShaders.add(shader);
        }

        if (!this.recompileFuture.isDone()) {
            return;
        }

        this.scheduleRecompile(0);
    }

    @Override
    public CompletableFuture<Void> method_25931(class_4045 preparationBarrier, class_3300 resourceManager, class_3695 preparationsProfiler, class_3695 reloadProfiler, Executor backgroundExecutor, Executor gameExecutor) {
        if (this.reloadFuture != null && !this.reloadFuture.isDone()) {
            return this.reloadFuture.thenCompose(preparationBarrier::method_18352);
        }
        return this.reloadFuture = this.recompileFuture.thenCompose(
                unused -> CompletableFuture.supplyAsync(() -> {
                            class_7654 lister = this.sourceSet.getShaderDefinitionLister();
                            Set<class_2960> shaderIds = lister.method_45113(resourceManager).keySet()
                                    .stream()
                                    .map(lister::method_45115)
                                    .collect(Collectors.toSet());
                            return this.prepare(resourceManager, shaderIds);
                        }, backgroundExecutor)
                        .thenCompose(preparationBarrier::method_18352)
                        .thenAcceptAsync(this::apply, gameExecutor));
    }

    @Override
    public String method_22322() {
        return this.getClass().getSimpleName() + " " + this.getSourceSet().getFolder();
    }

    /**
     * Recompiles all shaders in the background.
     *
     * @param resourceManager    The manager for resources. Shader files and definitions are loaded from this manager
     * @param backgroundExecutor The executor for preparation tasks
     * @param gameExecutor       The executor for applying the shaders
     * @return A future representing when shader compilation will be done
     */
    public CompletableFuture<Void> reload(class_3300 resourceManager, Executor backgroundExecutor, Executor gameExecutor) {
        VeilRenderer renderer = VeilRenderSystem.renderer();
        FramebufferManager framebufferManager = renderer.getFramebufferManager();
        PostProcessingManager postProcessingManager = renderer.getPostProcessingManager();

        return this.reloadFuture = CompletableFuture.allOf(
                this.reload(this, resourceManager, backgroundExecutor, gameExecutor),
                this.reload(framebufferManager, resourceManager, backgroundExecutor, gameExecutor),
                this.reload(postProcessingManager, resourceManager, backgroundExecutor, gameExecutor));
    }

    private CompletableFuture<Void> reload(class_3302 listener, class_3300 resourceManager, Executor backgroundExecutor, Executor gameExecutor) {
        return listener.method_25931(CompletableFuture::completedFuture,
                resourceManager,
                class_3694.field_16280,
                class_3694.field_16280,
                backgroundExecutor,
                gameExecutor);
    }

    /**
     * @return The current future for full shader reload status
     */
    public CompletableFuture<Void> getReloadFuture() {
        return this.reloadFuture;
    }

    /**
     * @return The current future for dirty shader recompilation status
     */
    public CompletableFuture<Void> getRecompileFuture() {
        return this.recompileFuture;
    }

    /**
     * Retrieves a readable name for a shader type. Supports all shader types instead of just vertex and fragment.
     *
     * @param type The GL enum for the type
     * @return The readable name or a hex value if the type is unknown
     */
    public static String getTypeName(int type) {
        String value = TYPES.get(type);
        return value != null ? value : "0x" + Integer.toHexString(type);
    }

    @Override
    public void close() {
        this.shaders.values().forEach(ShaderProgram::free);
        this.shaders.clear();
    }

    private record ReloadState(Map<class_2960, ProgramDefinition> definitions,
                               Map<class_2960, class_3298> shaderSources) {
    }
}
