package com.terraformersmc.terraform.boat.api.client;

import com.terraformersmc.terraform.boat.impl.client.TerraformBoatClientInitializer;
import it.unimi.dsi.fastutil.Pair;
import it.unimi.dsi.fastutil.objects.Object2ObjectOpenHashMap;
import net.minecraft.client.model.TexturedModelData;
import net.minecraft.client.render.entity.model.BoatEntityModel;
import net.minecraft.client.render.entity.model.ChestBoatEntityModel;
import net.minecraft.client.render.entity.model.ChestRaftEntityModel;
import net.minecraft.client.render.entity.model.EntityModelLayer;
import net.minecraft.client.render.entity.model.RaftEntityModel;
import net.minecraft.util.Identifier;
import net.minecraftforge.api.distmarker.Dist;
import net.minecraftforge.api.distmarker.OnlyIn;

import java.util.function.Supplier;

@OnlyIn(Dist.CLIENT)
@SuppressWarnings("deprecation")
/**
 * This class provides useful helpers for registering a {@linkplain com.terraformersmc.terraform.boat.api.TerraformBoatType Terraform boat} on the client.
 */
public final class TerraformBoatClientHelper {
	private TerraformBoatClientHelper() {
		return;
	}

	/**
	 * Gets the identifier of a {@linkplain EntityModelLayer model layer} for a boat of a given type.
	 * @param boatId the {@linkplain net.minecraft.util.Identifier identifier} of the {@linkplain com.terraformersmc.terraform.boat.api.TerraformBoatType boat}
	 * @param raft whether the boat is a raft
	 * @param chest whether the boat contains a chest
	 */
	private static Identifier getLayerId(Identifier boatId, boolean raft, boolean chest) {
		String prefix = raft ? (chest ? "chest_raft/" : "raft/") : (chest ? "chest_boat/" : "boat/");
		return new Identifier(boatId.getNamespace(), prefix + boatId.getPath());
	}

	/**
	 * Creates a {@linkplain EntityModelLayer model layer} for a boat of a given type.
	 * @param boatId the {@linkplain net.minecraft.util.Identifier identifier} of the {@linkplain com.terraformersmc.terraform.boat.api.TerraformBoatType boat}
	 * @param raft whether the boat is a raft
	 * @param chest whether the boat contains a chest
	 * 
	 * <pre>{@code
	 *     EntityModelLayer layer = TerraformBoatClientHelper.getLayer(new Identifier("examplemod", "mahogany"), false, false);
	 * }</pre>
	 */
	public static EntityModelLayer getLayer(Identifier boatId, boolean raft, boolean chest) {
		return new EntityModelLayer(getLayerId(boatId, raft, chest), "main");
	}

	private static Supplier<TexturedModelData> getTexturedModelDataProvider(boolean raft, boolean chest) {
		if (raft) {
			return chest ? ChestRaftEntityModel::getTexturedModelData : RaftEntityModel::getTexturedModelData;
		} else {
			return chest ? ChestBoatEntityModel::getTexturedModelData : BoatEntityModel::getTexturedModelData;
		}
	}

	/**
	 * Registers a {@linkplain EntityModelLayer model layer} for a boat of a given type.
	 * @param boatId the {@linkplain net.minecraft.util.Identifier identifier} of the {@linkplain com.terraformersmc.terraform.boat.api.TerraformBoatType boat}
	 * @param raft whether the boat is a raft
	 * @param chest whether the boat contains a chest
	 * 
	 * <pre>{@code
	 *     TerraformBoatClientHelper.registerModelLayer(new Identifier("examplemod", "mahogany"), false, false);
	 * }</pre>
	 */
	public static void registerModelLayer(Identifier boatId, boolean raft, boolean chest) {
		TerraformBoatClientInitializer.SUPPLIER_MAP.computeIfAbsent(boatId, b -> new Object2ObjectOpenHashMap<>()).put(chest, Pair.of(raft, getTexturedModelDataProvider(raft, chest)));
	}

	/**
	 * Registers {@linkplain EntityModelLayer model layers} for a given boat type.
	 * @param boatId the {@linkplain net.minecraft.util.Identifier identifier} of the {@linkplain com.terraformersmc.terraform.boat.api.TerraformBoatType boat type}
	 * @param raft whether the boat is a raft
	 * 
	 * <pre>{@code
	 *     TerraformBoatClientHelper.registerModelLayers(new Identifier("examplemod", "mahogany"), false);
	 * }</pre>
	 */
	public static void registerModelLayers(Identifier boatId, boolean raft) {
		registerModelLayer(boatId, raft, false);
		registerModelLayer(boatId, raft, true);
	}
}
