package com.samsthenerd.monthofswords.items;

import com.samsthenerd.monthofswords.SwordsMod;
import com.samsthenerd.monthofswords.registry.SwordsModStatusEffects;
import it.unimi.dsi.fastutil.ints.IntList;
import net.minecraft.ChatFormatting;
import net.minecraft.core.component.DataComponents;
import net.minecraft.item.*;
import net.minecraft.network.chat.Style;
import net.minecraft.resources.ResourceLocation;
import net.minecraft.server.level.ServerLevel;
import net.minecraft.util.*;
import net.minecraft.world.InteractionHand;
import net.minecraft.world.InteractionResultHolder;
import net.minecraft.world.damagesource.DamageSource;
import net.minecraft.world.effect.MobEffectInstance;
import net.minecraft.world.effect.MobEffects;
import net.minecraft.world.entity.Entity;
import net.minecraft.world.entity.EquipmentSlot;
import net.minecraft.world.entity.EquipmentSlotGroup;
import net.minecraft.world.entity.LivingEntity;
import net.minecraft.world.entity.Mob;
import net.minecraft.world.entity.SlotAccess;
import net.minecraft.world.entity.ai.attributes.AttributeModifier;
import net.minecraft.world.entity.ai.attributes.Attributes;
import net.minecraft.world.entity.player.Player;
import net.minecraft.world.entity.projectile.FireworkRocketEntity;
import net.minecraft.world.inventory.ClickAction;
import net.minecraft.world.inventory.Slot;
import net.minecraft.world.item.Item;
import net.minecraft.world.item.ItemStack;
import net.minecraft.world.item.Items;
import net.minecraft.world.item.SwordItem;
import net.minecraft.world.item.Tiers;
import net.minecraft.world.item.component.FireworkExplosion;
import net.minecraft.world.item.component.Fireworks;
import net.minecraft.world.level.Level;
import net.minecraft.world.phys.AABB;
import net.minecraft.world.phys.Vec3;
import java.util.List;
import java.util.function.UnaryOperator;

public class StealthSwordItem extends SwordtemberItem {

    private static final ResourceLocation STEALTH_SWORD_SNEAK_MODIFIER = ResourceLocation.fromNamespaceAndPath(SwordsMod.MOD_ID, "stealthswordsneak");
    private static final ResourceLocation STEALTH_SWORD_FALL_MODIFIER = ResourceLocation.fromNamespaceAndPath(SwordsMod.MOD_ID, "stealthswordfall");

    private static final Fireworks DEFAULT_FIREWORK = new Fireworks(1,
            List.of(new FireworkExplosion(
                    FireworkExplosion.Shape.SMALL_BALL,
                    IntList.of(ChatFormatting.BLACK.getColor()),
                    IntList.of(ChatFormatting.BLACK.getColor(), ChatFormatting.DARK_RED.getColor()),
                    false, false
            )
        )
    );

    public StealthSwordItem(Item.Properties itemSettings) {
        super(Tiers.IRON, itemSettings.attributes(
                SwordItem.createAttributes(Tiers.IRON, 3, -2.4f)
                        .withModifierAdded(
                                Attributes.SNEAKING_SPEED,
                                new AttributeModifier(STEALTH_SWORD_SNEAK_MODIFIER, 0.4, AttributeModifier.Operation.ADD_VALUE),
                                EquipmentSlotGroup.MAINHAND
                        ).withModifierAdded(
                                Attributes.SAFE_FALL_DISTANCE,
                                new AttributeModifier(STEALTH_SWORD_FALL_MODIFIER, 2, AttributeModifier.Operation.ADD_VALUE),
                                EquipmentSlotGroup.MAINHAND
                        )
        ));
    }

    @Override
    public InteractionResultHolder<ItemStack> use(Level world, Player user, InteractionHand hand) {
        boolean ready = !user.getCooldowns().isOnCooldown(this);
        ItemStack handStack = user.getItemInHand(hand);
        if(world instanceof ServerLevel sWorld && ready){
            if(!user.isShiftKeyDown()){
                ItemStack rocketStack = new ItemStack(Items.FIREWORK_ROCKET);
                rocketStack.set(
                        DataComponents.FIREWORKS,
                        user.getItemInHand(hand).getComponents().getOrDefault(DataComponents.FIREWORKS, DEFAULT_FIREWORK)
                );
                FireworkRocketEntity rocketEntity = new FireworkRocketEntity(world, rocketStack, null, user.getX(), user.getEyeY() - (double)0.15f, user.getZ(), true);
                rocketEntity.setDeltaMovement(user.getViewVector(0));
                sWorld.addFreshEntity(rocketEntity);

                // let the user escape by making all nearby mobs untarget them and all players have brief blindness
                for(Entity nearbyEnt : world.getEntities(user, new AABB(user.getEyePosition(), user.position()).inflate(user.entityInteractionRange()*2))){
                    if(nearbyEnt instanceof LivingEntity liveEnt){
                        liveEnt.addEffect(new MobEffectInstance(MobEffects.BLINDNESS, 20 * 5)); // give them blindness for 3 seconds
                    }
                    if(nearbyEnt instanceof Mob nearbyMob){

                        nearbyMob.addEffect(new MobEffectInstance(SwordsModStatusEffects.getEffect(SwordsModStatusEffects.SMOKE_BOMBED), 20 * 5));
                    }
                }
                handStack.hurtAndBreak(2, user, hand == InteractionHand.MAIN_HAND ? EquipmentSlot.MAINHAND : EquipmentSlot.OFFHAND);
            }
            user.addEffect(new MobEffectInstance(MobEffects.INVISIBILITY, 30 * 20));
            user.addEffect(new MobEffectInstance(MobEffects.MOVEMENT_SPEED, 15 * 20, 2));
            user.addEffect(new MobEffectInstance(MobEffects.JUMP, 15 * 20, 2));
            user.getCooldowns().addCooldown(this, 20 * 30); // 30 sec cooldown - to encourage more frequent retreats
            return InteractionResultHolder.success(handStack);
        }
        return ready ? InteractionResultHolder.success(handStack) : InteractionResultHolder.pass(handStack);
    }

    @Override
    public float getAttackDamageBonus(Entity target, float baseAttackDamage, DamageSource damageSource) {
        Entity attacker = damageSource.getEntity();
        float extraDamage = super.getAttackDamageBonus(target, baseAttackDamage, damageSource);
        if(!(attacker instanceof LivingEntity liveAttacker && liveAttacker.level() instanceof ServerLevel sWorld)) return extraDamage;
        boolean wasSuperSneakAttack = false;
        if(liveAttacker.hasEffect(MobEffects.INVISIBILITY)){
            if(liveAttacker instanceof Player playerAttacker && playerAttacker.getCooldowns().isOnCooldown(this)){
                wasSuperSneakAttack = true;
                extraDamage += 0.75f;
            }
            extraDamage += 0.5f;
            liveAttacker.removeEffect(MobEffects.INVISIBILITY);
        }
        Vec3 targetLook = target.getViewVector(0).multiply(1, 1, 0).normalize();
        Vec3 attackerToTargetVec = target.position().subtract(attacker.position()).normalize();
        double dotProd = targetLook.dot(attackerToTargetVec);
        if(dotProd > 0){ // sneak attack !
            extraDamage += 1f;
            if(wasSuperSneakAttack && target instanceof LivingEntity liveTarget){
                // TODO: make this give an achievement for sneak attacking ?
                liveTarget.addEffect(new MobEffectInstance(MobEffects.BLINDNESS, 20 * 5)); // give them blindness for 5 seconds
            }
        }
        if(attacker.isShiftKeyDown()){
            extraDamage += 0.25f;
        }
        return extraDamage;
    }

    @Override
    public boolean overrideOtherStackedOnMe(ItemStack stack, ItemStack otherStack, Slot slot, ClickAction clickType, Player player, SlotAccess cursorStackReference) {
        if(otherStack.getItem().equals(Items.FIREWORK_ROCKET)){
            stack.set(DataComponents.FIREWORKS, otherStack.get(DataComponents.FIREWORKS));
            otherStack.shrink(1);
            return true;
        }
        return false;
    }
//
//    @Override
//    public void appendTooltip(ItemStack stack, TooltipContext context, List<Text> tooltip, TooltipType type) {
//        if(SwordtemberItem.hasShiftSafe()){
//            MutableText infoText = Text.translatable(stack.getTranslationKey() + ".tooltip");
//            infoText.setStyle(getSwordTooltipStyleModifier().apply(Style.EMPTY.withItalic(true)));
//            MutableText infoText2 = Text.translatable(stack.getTranslationKey() + ".tooltip.2");
//            infoText2.setStyle(getSwordTooltipStyleModifier().apply(Style.EMPTY.withItalic(true)));
//            MutableText infoText3 = Text.translatable(stack.getTranslationKey() + ".tooltip.3");
//            infoText3.setStyle(getSwordTooltipStyleModifier().apply(Style.EMPTY.withItalic(true)));
//            tooltip.add(infoText);
//            tooltip.add(Text.literal(""));
//            tooltip.add(infoText2);
//            tooltip.add(Text.literal(""));
//            tooltip.add(infoText3);
//        } else {
//            MutableText shiftMsg = Text.translatable("monthofswords.tooltip.shiftmsg");
//            shiftMsg.setStyle(getSwordTooltipStyleModifier().apply(Style.EMPTY.withItalic(true)));
//            tooltip.add(shiftMsg);
//        }
//        super.appendTooltip(stack, context, tooltip, type);
//    }

    public UnaryOperator<Style> getSwordTooltipStyleModifier() {
        return (style) -> style.withColor(ChatFormatting.DARK_RED);
    }
}