package com.samsthenerd.monthofswords.utils;

import com.mojang.serialization.Codec;
import com.mojang.serialization.codecs.RecordCodecBuilder;
import com.samsthenerd.monthofswords.registry.SwordsModComponents;
import com.samsthenerd.monthofswords.registry.SwordsModLoot;
import com.samsthenerd.monthofswords.tooltips.RecipeTooltipData;
import com.samsthenerd.monthofswords.utils.Description.AcquisitionDesc.CraftingDesc;
import com.samsthenerd.monthofswords.utils.Description.AcquisitionDesc.LootDropDesc;
import com.samsthenerd.monthofswords.utils.Description.AcquisitionDesc.SpecificText;
import dev.architectury.platform.Platform;
import dev.architectury.registry.registries.RegistrySupplier;
import dev.architectury.utils.Env;
import java.util.*;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.function.UnaryOperator;
import net.minecraft.ChatFormatting;
import net.minecraft.client.gui.screens.Screen;
import net.minecraft.client.resources.language.I18n;
import net.minecraft.network.RegistryFriendlyByteBuf;
import net.minecraft.network.chat.Component;
import net.minecraft.network.chat.MutableComponent;
import net.minecraft.network.codec.ByteBufCodecs;
import net.minecraft.network.codec.StreamCodec;
import net.minecraft.resources.ResourceKey;
import net.minecraft.resources.ResourceLocation;
import net.minecraft.util.Tuple;
import net.minecraft.world.item.Item;
import net.minecraft.world.item.Item.TooltipContext;
import net.minecraft.world.item.ItemStack;
import net.minecraft.world.item.TooltipFlag;
import net.minecraft.world.level.storage.loot.LootTable;

// for now just assume this stuff is only called on the client ig?
public record Description(RegistrySupplier<? extends Item> item, List<SwordPower> powers, List<AcquisitionDesc> acqDescs,
                          int textColor){


    public List<Component> getSummaryTooltip(){
//        var briefTexts = getContinuousText(makeLangPatternProvider(item.get().getTranslationKey() + ".brief", List.of()))
//            .stream().map(t -> applyColor(t).formatted(Formatting.ITALIC)).toList();
        List<Component> resTT = new ArrayList<>();
            resTT.addAll(powers.stream().map(p -> p.getPowerTitle(this)).toList());
            if(!resTT.isEmpty()){
                resTT.add(Component.literal(""));
                resTT.add(applyColor(Component.translatable("monthofswords.tooltip.shiftpowers")).withStyle(ChatFormatting.ITALIC));
            }
        return resTT;
    }

    public List<Component> getPowerTooltip(){
        List<Component> powTTs = new ArrayList<>();
        for(var pow : powers){
            powTTs.addAll(pow.getPowerTooltip(this));
            powTTs.add(Component.literal(""));
        }
        if(!powTTs.isEmpty()) powTTs.removeLast();
        return powTTs;
    }

    public List<Component> getAcquisitionTooltip(){
        List<Component> acqTTs = new ArrayList<>();
        for(var acqd : acqDescs){
            acqTTs.addAll(acqd.getAcqTooltip().stream().map(this::applyColor).toList());
            acqTTs.add(Component.literal(""));
        }
        return acqTTs;
    }

    public List<Component> getTooltipFull(ItemStack stack, TooltipContext context, TooltipFlag type){
        var descData = stack.get(SwordsModComponents.ITEM_DESCRIPTION_DATA);
        if(descData == null){
            // normal !
            if (hasShiftSafe()) {
                return getPowerTooltip();
            } else {
                return getSummaryTooltip();
            }
        } else {
            // in calendar
            List<Component> tt = new ArrayList<>();
            if(descData.hintMode){
                tt.addAll(getAcquisitionTooltip());
                tt.add(applyColor(Component.translatable("monthofswords.descriptionutil.switchtopower")).withStyle(ChatFormatting.ITALIC));
            } else {
                if (hasShiftSafe() || descData.showFull()) {
                    tt.addAll(getPowerTooltip());
                } else {
                    tt.addAll(getSummaryTooltip());
                }
                if(!descData.showFull()){
                    tt.add(Component.literal(""));
                    tt.add(applyColor(Component.translatable("monthofswords.descriptionutil.switchtohint")).withStyle(ChatFormatting.ITALIC));
                }
            }
            return tt;
        }
    }

    public static Description forItem(RegistrySupplier<? extends Item> item){
        return new Description(item, new ArrayList<>(), new ArrayList<>(), 0xFFFFFF);
    }

    public Description withPower(SwordPower... power){
        List<SwordPower> newPowers = new ArrayList<>(powers);
        newPowers.addAll(Arrays.asList(power));
        return new Description(item, newPowers, acqDescs, textColor);
    }

    public Description withAcquisitionDesc(AcquisitionDesc... descs){
        List<AcquisitionDesc> newAcqs = new ArrayList<>(acqDescs);
        newAcqs.addAll(Arrays.asList(descs));
        return new Description(item, powers, newAcqs, textColor);
    }

    public Description withLootAcqDesc(){
        return withAcquisitionDesc(new LootDropDesc(SwordsModLoot.LOOT_LISTS.get(item.getId()).stream().toList()));
    }

    public Description withMatchingRecipe(){
        return withAcquisitionDesc(new CraftingDesc(item.getId()));
    }

    public Description withSpecificAcqDesc(){
        return withAcquisitionDesc(new SpecificText(item.get().getDescriptionId()));
    }

    public Description withTextColor(int color){
        return new Description(item, powers, acqDescs, color);
    }

    public Description withTextColor(ChatFormatting fm){
        return new Description(item, powers, acqDescs, fm.isColor() ? fm.getColor() : textColor);
    }

    public Description finalize(Consumer<Description> consumer){
        consumer.accept(this);
        return this;
    }

    // most condensed way to make it show only when create is installed lol
    public Description conditionally(UnaryOperator<Description> op, boolean condition){
        return condition ? op.apply(this) : this;
    }

    public MutableComponent applyColor(Component t){
        return t.copy().withColor(textColor);
    }

    /**
     * clientside helper for getting a sequence of translation entries whose length is determined by the available
     * translations, with support for ranked order key choice.
     *
     * The sequence will end when no lang keys have translations or any of them are translated as just "/endseq/"
     *
     * This is mostly overengineered for the purpose of modpack devs being able to change stuff if they want,
     * will that ever happen? prob not
     *
     * @param langPatterns a function that takes an integer and returns a ranked list of possible lang keys
     * @param args args to provide to each line of text
     * @return the sequence of texts
     */
    public static List<Component> getContinuousText(Function<Integer, List<String>> langPatterns, Object... args){
        int i = 0;
        boolean keepGoing = true;
        List<Component> ts = new ArrayList<>();
        while(keepGoing){
            var keys = langPatterns.apply(i);
            keepGoing = false;
            for(String k : keys){
                if(!I18n.exists(k)){
                    continue;
                }
                String str = I18n.get(k, args);
                if(str.equals("/endseq/")) break;
                ts.add(Component.translatable(k, args));
                keepGoing = true;
                break;
            }
            i++;
        }
        return ts;
    }

    public static Function<Integer, List<String>> makeLangPatternProvider(String base, List<String> flags){
        return i -> {
            List<String> keys = new ArrayList<>();
            for(String f : flags){
                if(i == 0) keys.add(base + "." + f);
                keys.add(base + "." + i + "." + f);
            }
            if(i == 0) keys.add(base);
            keys.add(base + "." + i);
            return keys;
        };
    }

    public record SwordPower(String name, Component powerType, Optional<Integer> cooldown){
        public List<Component> getPowerTooltip(Description desc){
            String powerTitleKey = desc.item.get().getDescriptionId() + ".power." + name;
            Component fullPowerTitle = getPowerTitle(desc);
            List<Component> powerText = new ArrayList<>();
            powerText.add(fullPowerTitle);
            var powerDescs = getContinuousText(makeLangPatternProvider(powerTitleKey + ".desc", List.of()));
            cooldown.ifPresent(cd -> powerDescs.add(Component.translatable("monthofswords.descriptionutil.powercooldown", Description.getFormattedTime(cd))));
            powerText.addAll(
                powerDescs.stream().map(
                    t -> Component.literal(" ")
                        .append(t)
                        .withStyle(ChatFormatting.ITALIC)
                        .withColor(desc.textColor))
                .toList()
            );
            return powerText;
        }

        public Component getPowerTitle(Description desc){
            String powerTitleKey = desc.item.get().getDescriptionId() + ".power." + name;
            var powerTitle = Component.translatableWithFallback(powerTitleKey, "");
            Component fullPowerTitle;
            if(I18n.exists(powerTitleKey)){
                fullPowerTitle = Component.translatable("monthofswords.descriptionutil.powertitleformat", powerTitle, powerType);
            } else {
                fullPowerTitle = powerType;
            }
            return fullPowerTitle.copy().withStyle(ChatFormatting.BOLD, ChatFormatting.UNDERLINE).withColor(desc.textColor());
        }
    }

    public static final Component PASSIVE_POWER = Component.translatable("monthofswords.descriptionutil.powertype.passivepower");
    public static final Component HIT_POWER = Component.translatable("monthofswords.descriptionutil.powertype.hitpower");
    public static final Component USE_POWER = Component.translatable("monthofswords.descriptionutil.powertype.usepower");
    public static final Component CHARGE_USE_POWER = Component.translatable("monthofswords.descriptionutil.powertype.chargeusepower");
    public static final Component HOLD_POWER = Component.translatable("monthofswords.descriptionutil.powertype.holdusepower");
    public static final Component ACTION_POWER = Component.translatable("monthofswords.descriptionutil.powertype.actionpower", Component.keybind("key.swordsmod.action"));
    public static final Component SWING_POWER = Component.translatable("monthofswords.descriptionutil.powertype.swingpower");
    public static final Component INV_USE = Component.translatable("monthofswords.descriptionutil.powertype.invusepower");


    public interface AcquisitionDesc {

        List<Component> getAcqTooltip();

        record LootDropDesc(List<Tuple<ResourceKey<LootTable>, Float>> tableChances) implements AcquisitionDesc{
            @Override
            public List<Component> getAcqTooltip() {
                List<Component> tt = new ArrayList<>();
                tt.add(Component.translatable("monthofswords.descriptionutil.acq.title.loot"));
                for(var tc : tableChances){
                    MutableComponent t = Component.literal(" ");
                    Component tableText = getLootTableName(tc.getA());
                    float roundedVal = Math.round(tc.getB()*10000)/100f;
                    t.append(Component.translatable("monthofswords.descriptionutil.acq.loot", roundedVal, tableText));
                    tt.add(t);
                }
                return tt;
            }

            // idk why EMI loot has their stuff prefixed like this
            public static Map<String, String> EMI_LOOT_TYPE_LOOKUP = new HashMap<>(Map.of(
                "chests", "chest",
                "spawners", "chest",
                "dispensers", "chest"
            ));

            public static Component getLootTableName(ResourceKey<LootTable> lootTableKey){
                ResourceLocation lootId = lootTableKey.location();
                String sensibleKey = "loot." + lootId.getNamespace() + "." + lootId.getPath();
                // descriptions could be just sensibleKey with .desc or .description suffix ?
                if(I18n.exists(sensibleKey)){
                    return Component.translatable(sensibleKey);
                }
                String lootType = lootId.getPath().split("/")[0];
                String lootTypeEMI = EMI_LOOT_TYPE_LOOKUP.getOrDefault(lootType, lootType);
                String emiKey = "emi_loot." + lootTypeEMI + "." + lootId;
                if(I18n.exists(emiKey)){
                    return Component.translatable(emiKey);
                }
                return Component.literal(lootId.toString());
            }
        }

        record CraftingDesc(ResourceLocation recId) implements AcquisitionDesc{
            @Override
            public List<Component> getAcqTooltip() {
                return List.of();
            }
        }

        record SpecificText(String baseItemKey) implements AcquisitionDesc{
            @Override
            public List<Component> getAcqTooltip() {
                return getContinuousText(makeLangPatternProvider( baseItemKey + ".acquisition", List.of()));
            }
        }
    }

    public static int[] TIME_HIERARCHY = {24 * 60 * 60 * 20, 60*60*20, 60*20, 20};
    public static String[] TIME_HIERARCHY_LABELS = {"day", "hour", "min", "sec"};

    public static Component getFormattedTime(int tickCount){
//        List<Integer> times = new ArrayList<>();
        MutableComponent timeText = Component.empty();
        boolean modified = false;
        for(int i = 0; i < 3; i++){
//            times.add(timeUnit);
            int timeUnit = tickCount / TIME_HIERARCHY[i];
            if(timeUnit != 0){
                timeText.append(
                    Component.translatable("monthofswords.descriptionutil.timelabel." + TIME_HIERARCHY_LABELS[i],
                        timeUnit));
                modified = true;
            }

            tickCount %= TIME_HIERARCHY[i];
        }
        double secs = tickCount / 20.0;
        if(secs != 0 || !modified){
            timeText.append(Component.translatable("monthofswords.descriptionutil.timelabel.sec", secs));
        }
        return timeText;
    }

    public record DescriptionItemComponent(boolean hintMode, Optional<RecipeTooltipData> ttData, boolean showFull){
        public static final Codec<DescriptionItemComponent> CODEC = RecordCodecBuilder.create(instance ->
            instance.group(
                Codec.BOOL.fieldOf("hintMode").forGetter(DescriptionItemComponent::hintMode),
                RecipeTooltipData.CODEC.optionalFieldOf("ttData").forGetter(DescriptionItemComponent::ttData),
                Codec.BOOL.fieldOf("showFull").forGetter(DescriptionItemComponent::showFull)
            ).apply(instance, DescriptionItemComponent::new)
        );

        public static final StreamCodec<RegistryFriendlyByteBuf, DescriptionItemComponent> PACKET_CODEC = StreamCodec.composite(
            ByteBufCodecs.BOOL, DescriptionItemComponent::hintMode,
            RecipeTooltipData.PACKET_CODEC.apply(ByteBufCodecs::optional), DescriptionItemComponent::ttData,
            ByteBufCodecs.BOOL, DescriptionItemComponent::showFull,
            DescriptionItemComponent::new
        );
    }

    // returns true if shift is down or if it's on the server
    // meant to be used for tooltips to not break polydex
    public static boolean hasShiftSafe(){
        return Platform.getEnvironment() == Env.SERVER || Screen.hasShiftDown();
    }
}
