package com.samsthenerd.monthofswords.items;

import com.samsthenerd.monthofswords.SwordsMod;
import net.minecraft.core.BlockPos;
import net.minecraft.core.Direction;
import net.minecraft.item.*;
import net.minecraft.network.chat.Style;
import net.minecraft.resources.ResourceKey;
import net.minecraft.server.level.ServerLevel;
import net.minecraft.server.level.ServerPlayer;
import net.minecraft.tags.BlockTags;
import net.minecraft.world.entity.Entity;
import net.minecraft.world.entity.player.Player;
import net.minecraft.world.item.Item;
import net.minecraft.world.item.ItemStack;
import net.minecraft.world.item.Items;
import net.minecraft.world.item.SwordItem;
import net.minecraft.world.item.Tier;
import net.minecraft.world.item.crafting.Ingredient;
import net.minecraft.world.level.Level;
import net.minecraft.world.level.block.Blocks;
import net.minecraft.world.level.block.state.BlockState;
import net.minecraft.world.level.border.WorldBorder;
import net.minecraft.world.level.dimension.DimensionType;
import net.minecraft.world.level.portal.DimensionTransition;
import org.jetbrains.annotations.Nullable;

import java.util.function.UnaryOperator;

public class PortalSwordItem extends SwordtemberItem implements SwordActionHaverServer {
    public static final Tier OBSIDIAN_MATERIAL = new ClassyToolMaterial(750, 7f, 2.5f,
        BlockTags.INCORRECT_FOR_DIAMOND_TOOL, 16, () -> Ingredient.of(Items.OBSIDIAN));

    public PortalSwordItem(Item.Properties itemSettings) {
        super(OBSIDIAN_MATERIAL, itemSettings.attributes(
            SwordItem.createAttributes(OBSIDIAN_MATERIAL, 3, -2.4f))
        );
    }

    @Override
    public UnaryOperator<Style> getSwordTooltipStyleModifier(){
        return (style) -> style.withColor(0xaf4fe3);
    }

    @Override
    public boolean doSwordAction(Player player, ItemStack swordStack){
        if(!(player.level() instanceof ServerLevel sWorld)
        || !SwordsMod.canBeDestructive(player, null)) return false;
        DimensionTransition teleportTarget = createTeleportTarget(sWorld, player, player.blockPosition());
        if (teleportTarget != null) {
            ServerLevel otherWorld = teleportTarget.newLevel();
            if (sWorld.getServer().isLevelEnabled(otherWorld)
                && (otherWorld.dimension() == sWorld.dimension()
                || player.canChangeDimensions(sWorld, otherWorld))) {
                player.changeDimension(teleportTarget);
            }
        }
        return true;
    }

    @Nullable
    public static DimensionTransition createTeleportTarget(ServerLevel initialWorld, Entity entity, BlockPos pos) {
        ResourceKey<Level> registryKey = initialWorld.dimension() == Level.NETHER ? Level.OVERWORLD : Level.NETHER;
        ServerLevel otherWorld = initialWorld.getServer().getLevel(registryKey);
        if (otherWorld == null) { return null; }
        if(!(entity instanceof ServerPlayer player)) return null;
        boolean isNether = otherWorld.dimension() == Level.NETHER;
        WorldBorder worldBorder = otherWorld.getWorldBorder();
        double coordScale = DimensionType.getTeleportationScale(initialWorld.dimensionType(), otherWorld.dimensionType());
        BlockPos scaledPos = worldBorder.clampToBounds(entity.getX() * coordScale,
            Math.clamp(entity.getY(), otherWorld.getMinBuildHeight()+1, otherWorld.getMaxBuildHeight()-1),
            entity.getZ() * coordScale);

        DimensionTransition.PostDimensionTransition postDimensionTransition = DimensionTransition.PLAY_PORTAL_SOUND
            .then(DimensionTransition.PLACE_PORTAL_TICKET).then(entTeleported -> {
                // try to make it safer if possible.
                BlockState baseState = otherWorld.getBlockState(scaledPos);
                if(baseState.isSuffocating(otherWorld, scaledPos)
                    && otherWorld.mayInteract(player, scaledPos)
                    && baseState.getBlock().defaultDestroyTime() > 0){
                    otherWorld.destroyBlock(scaledPos, true, player);
                }
                // if you're scaled up that's on you.
                BlockPos headPos = scaledPos.relative(Direction.UP);
                BlockState headState = otherWorld.getBlockState(headPos);
                if(headState.isSuffocating(otherWorld, headPos)
                    && otherWorld.mayInteract(player, headPos)
                    && headState.getBlock().defaultDestroyTime() > 0){
                    otherWorld.destroyBlock(headPos, true, player);
                }
                // place a block under you if needed.
                BlockPos underPos = scaledPos.relative(Direction.DOWN);
                if(otherWorld.getBlockState(underPos).isAir() && otherWorld.mayInteract(player, underPos)){
                    otherWorld.setBlockAndUpdate(underPos, isNether ? Blocks.NETHERRACK.defaultBlockState() : Blocks.DIRT.defaultBlockState());
                }
                // any other safety stuff ? maybe fire resistance ?
            });
        return new DimensionTransition(otherWorld, scaledPos.getCenter(), entity.getDeltaMovement(), entity.getYRot(), entity.getXRot(), postDimensionTransition);
    }
}
