package com.samsthenerd.monthofswords.items;

import I;
import com.samsthenerd.monthofswords.registry.SwordsModStatusEffects;
import com.samsthenerd.monthofswords.utils.FollowLeaderGoal;
import org.jetbrains.annotations.Nullable;

import java.util.Comparator;
import java.util.List;
import java.util.function.Predicate;
import java.util.function.UnaryOperator;
import net.minecraft.ChatFormatting;
import net.minecraft.network.chat.Style;
import net.minecraft.server.level.ServerLevel;
import net.minecraft.server.level.ServerPlayer;
import net.minecraft.tags.EntityTypeTags;
import net.minecraft.world.InteractionHand;
import net.minecraft.world.InteractionResultHolder;
import net.minecraft.world.damagesource.DamageSource;
import net.minecraft.world.effect.MobEffectInstance;
import net.minecraft.world.entity.Entity;
import net.minecraft.world.entity.EntityType;
import net.minecraft.world.entity.LivingEntity;
import net.minecraft.world.entity.OwnableEntity;
import net.minecraft.world.entity.ai.goal.Goal;
import net.minecraft.world.entity.ai.goal.RandomStrollGoal;
import net.minecraft.world.entity.monster.Monster;
import net.minecraft.world.entity.monster.Skeleton;
import net.minecraft.world.entity.monster.Zombie;
import net.minecraft.world.entity.player.Player;
import net.minecraft.world.item.ItemStack;
import net.minecraft.world.item.SwordItem;
import net.minecraft.world.item.Tiers;
import net.minecraft.world.item.UseAnim;
import net.minecraft.world.level.Level;
import net.minecraft.world.phys.AABB;
import net.minecraft.world.phys.Vec3;

public class NecromancerSwordItem extends SwordtemberItem {
//    public static final ToolMaterial NECROMANCER_MATERIAL = new ClassyToolMaterial(500, 5f, 3f,
//        BlockTags.INCORRECT_FOR_STONE_TOOL, 15, () -> Ingredient.ofItems(Items.STONE));

    public NecromancerSwordItem(Properties itemSettings) {
        super(Tiers.NETHERITE, itemSettings.attributes(
            SwordItem.createAttributes(Tiers.NETHERITE, 3, -2.4f))
        );
    }

    @Override
    public void inventoryTick(ItemStack stack, Level world, Entity entity, int slot, boolean selected) {
        if(selected && entity instanceof LivingEntity livEnt && !world.isClientSide()){
            livEnt.addEffect(new MobEffectInstance(SwordsModStatusEffects.getEffect(SwordsModStatusEffects.NECROMANCER),
                20*15));
        }
    }

    public static List<Entity> findEnemies(Vec3 nearPos, ServerPlayer owner){
        var enemyList = owner.serverLevel().getEntities(null,
            AABB.ofSize(nearPos, 16, 8, 16),
            ent -> ent instanceof Monster hent && hent.isAlive() &&
                (hent.getTarget() == owner || (hent.getTarget() instanceof OwnableEntity tmbl && tmbl.getOwner() == owner)));
        enemyList.sort(Comparator.comparing(ent -> ent.position().distanceTo(nearPos)));
        return enemyList;
    }

    @Override
    public void releaseUsing(ItemStack stack, Level world, LivingEntity user, int remainingUseTicks) {
        if(user instanceof ServerPlayer sPlayer){
            int timeHeld = 72000 - remainingUseTicks;
            int fishCount = Math.clamp(timeHeld / (5), 1, 6); // fish every quarter second
            sPlayer.getCooldowns().addCooldown(this, 20*fishCount*3);
            var enemyList = findEnemies(sPlayer.position(), sPlayer);
            for(int i = 0; i < fishCount; i++){
                var undead = makeRandomUndead(sPlayer);
                undead.setPos(sPlayer.position());
                sPlayer.serverLevel().addFreshEntity(undead);
                if(!enemyList.isEmpty()){
                    var target = enemyList.get(sPlayer.getRandom().nextIntBetweenInclusive(0, Math.min(enemyList.size()-1, 3)));
                    undead.setTarget((Monster) target);
                }
            }
        }
    }

    @Override
    public UseAnim getUseAnimation(ItemStack stack) {
        return UseAnim.BOW;
    }

    @Override
    public int getUseDuration(ItemStack stack, LivingEntity user) {
        if(user instanceof Player player && player.getCooldowns().isOnCooldown(this)) return 0;
        return 72000;
    }

    @Override
    public boolean useOnRelease(ItemStack stack) {
        return true;
    }

    @Override
    public InteractionResultHolder<ItemStack> use(Level world, Player user, InteractionHand hand) {
        ItemStack itemStack = user.getItemInHand(hand);
        user.startUsingItem(hand);
        return InteractionResultHolder.consume(itemStack);
    }

    @Override
    public float getAttackDamageBonus(Entity target, float baseAttackDamage, DamageSource damageSource) {
        if(target instanceof LivingEntity livEnt && livEnt.getType().is(EntityTypeTags.UNDEAD)){
            return -100;
        }
        return 0;
    }

    @Override
    public boolean hurtEnemy(ItemStack stack, LivingEntity target, LivingEntity attacker) {
        if(attacker.level() instanceof ServerLevel sWorld
        && !target.getType().is(EntityTypeTags.UNDEAD)){
            var bugsList = sWorld.getEntities(target,
                AABB.ofSize(target.position(), 16, 8, 16),
                ent -> ent.getType().is(EntityTypeTags.UNDEAD) && ent instanceof Monster);
            for(var bug : bugsList){
                if(bug instanceof Monster hostEnt){
                    var existTarget = hostEnt.getTarget();
                    if(existTarget == null || !existTarget.isAlive()
                        || target.distanceTo(bug) - existTarget.distanceTo(bug) < 2 || attacker.getRandom().nextFloat() < 0.2){
                        hostEnt.setTarget(target);
                    }
                }
            }
        }
        return super.hurtEnemy(stack, target, attacker);
    }


    @Override
    public UnaryOperator<Style> getSwordTooltipStyleModifier(){
        return (style) -> style.withColor(ChatFormatting.GRAY);
    }

    // not really a better way to do this huh
    public static Monster makeRandomUndead(ServerPlayer sPlayer){
        if(sPlayer.getRandom().nextBoolean()){
            return new Zombie(EntityType.ZOMBIE, sPlayer.level()){

                @Override
                public void setTarget(@Nullable LivingEntity target) {
                    if(target == sPlayer) return; // friend
                    super.setTarget(target);
                }

                @Override
                public void tick() {
                    LivingEntity tgt = this.getTarget();
                    if(tgt != null && !tgt.isAlive()){
                        var enemyList = findEnemies(this.position(), sPlayer);
                        if(!enemyList.isEmpty()){
                            var target = enemyList.get(sPlayer.getRandom().nextIntBetweenInclusive(0, Math.min(enemyList.size()-1, 3)));
                            this.setTarget((Monster) target);
                        } else {
                            this.setTarget(null);
                        }
                    }
                    if(tgt == null){
                        var lastFight = Math.max(this.getLastHurtByMobTimestamp(), this.getLastHurtMobTimestamp());
                        var peaceTime = this.tickCount - lastFight;
                        if(peaceTime > 20 * 20 && lastFight != 0 && sPlayer.getRandom().nextFloat() < 0.003){
                            this.spawnAnim();
                            this.discard();
                        }
                    }
                    super.tick();
                }

                @Override
                public boolean killedEntity(ServerLevel world, LivingEntity other) {
                    return super.killedEntity(world, other);
                }

                @Override
                protected void registerGoals() {
                    this.goalSelector.addGoal(4, new FollowLeaderGoal(this, sPlayer, 1.0, 4.0F, 2.0F));
                    super.registerGoals();
                    Predicate<Goal> isWander = goal -> goal instanceof RandomStrollGoal;
                    this.goalSelector.removeAllGoals(isWander);
                }
            };
        } else {
            return new Skeleton(EntityType.SKELETON, sPlayer.level()){

                @Override
                public void setTarget(@Nullable LivingEntity target) {
                    if(target == sPlayer) return; // friend
                    super.setTarget(target);
                }

                @Override
                public void tick() {
                    LivingEntity tgt = this.getTarget();
                    if(tgt != null && !tgt.isAlive()){
                        var enemyList = findEnemies(this.position(), sPlayer);
                        if(!enemyList.isEmpty()){
                            var target = enemyList.get(sPlayer.getRandom().nextIntBetweenInclusive(0, Math.min(enemyList.size()-1, 3)));
                            this.setTarget((Monster) target);
                        } else {
                            this.setTarget(null);
                        }
                    }
                    if(tgt == null){
                        var lastFight = Math.max(this.getLastHurtByMobTimestamp(), this.getLastHurtMobTimestamp());
                        var peaceTime = this.tickCount - lastFight;
                        if(peaceTime > 20 * 20 && lastFight != 0 && sPlayer.getRandom().nextFloat() < 0.003){
                            this.spawnAnim();
                            this.discard();
                        }
                    }
                    super.tick();
                }

                @Override
                public boolean killedEntity(ServerLevel world, LivingEntity other) {
                    return super.killedEntity(world, other);
                }

                @Override
                protected void registerGoals() {
                    this.goalSelector.addGoal(4, new FollowLeaderGoal(this, sPlayer, 1.0, 4.0F, 2.0F));
                    super.registerGoals();
                    Predicate<Goal> isWander = goal -> goal instanceof RandomStrollGoal;
                    this.goalSelector.removeAllGoals(isWander);
                }
            };
        }
    }
}
