package com.samsthenerd.monthofswords.items;

import com.mojang.serialization.Codec;
import com.mojang.serialization.codecs.RecordCodecBuilder;
import com.samsthenerd.monthofswords.registry.SwordsModComponents;
import net.minecraft.ChatFormatting;
import net.minecraft.core.UUIDUtil;
import net.minecraft.core.registries.BuiltInRegistries;
import net.minecraft.item.*;
import net.minecraft.network.RegistryFriendlyByteBuf;
import net.minecraft.network.chat.Component;
import net.minecraft.network.codec.ByteBufCodecs;
import net.minecraft.network.codec.StreamCodec;
import net.minecraft.resources.ResourceLocation;
import net.minecraft.server.level.ServerPlayer;
import net.minecraft.tags.BlockTags;
import net.minecraft.world.damagesource.DamageSource;
import net.minecraft.world.entity.Entity;
import net.minecraft.world.entity.EntityType;
import net.minecraft.world.entity.LivingEntity;
import net.minecraft.world.item.Item;
import net.minecraft.world.item.ItemStack;
import net.minecraft.world.item.Items;
import net.minecraft.world.item.SwordItem;
import net.minecraft.world.item.Tier;
import net.minecraft.world.item.TooltipFlag;
import net.minecraft.world.item.crafting.Ingredient;
import java.util.List;
import java.util.Optional;
import java.util.UUID;

public class CrystalSwordItem extends SwordtemberItem {

    public static final Tier AMETHYST_CRYSTAL_MATERIAL = new ClassyToolMaterial(500, 5f, 2f,
        BlockTags.INCORRECT_FOR_IRON_TOOL, 16, () -> Ingredient.of(Items.COPPER_INGOT));

    public CrystalSwordItem(Item.Properties itemSettings) {
        super(AMETHYST_CRYSTAL_MATERIAL, itemSettings.attributes(
            SwordItem.createAttributes(AMETHYST_CRYSTAL_MATERIAL, 3, -2.4f))
        );
    }

    @Override
    public void appendHoverText(ItemStack stack, TooltipContext context, List<Component> tooltip, TooltipFlag type) {
        var resCompOpt = Optional.ofNullable(stack.get(SwordsModComponents.RESONANCE_DATA));
        resCompOpt.flatMap(ResonatingComponent::getDamageStateText).ifPresent(tooltip::add);
        super.appendHoverText(stack, context, tooltip, type);
    }

    @Override
    public float getAttackDamageBonus(Entity target, float baseAttackDamage, DamageSource damageSource) {
        ItemStack stack = damageSource.getWeaponItem();
        float baseDamage = super.getAttackDamageBonus(target, baseAttackDamage, damageSource);
        if(stack == null || stack.isEmpty()) return baseDamage;
        ResonatingComponent resComp = stack.get(SwordsModComponents.RESONANCE_DATA);
        if(resComp == null) return baseDamage;
        if(EntityType.getKey(target.getType()).equals(resComp.entityType)) return resComp.extraDamage + baseDamage;
        return baseDamage;
    }

    @Override
    public void postHurtEnemy(ItemStack stack, LivingEntity target, LivingEntity attacker) {
        var resCompOpt = Optional.ofNullable(stack.get(SwordsModComponents.RESONANCE_DATA));
        ResourceLocation targetType = EntityType.getKey(target.getType());
        ResonatingComponent resComp = resCompOpt.map(resCompOld -> {
            float newDamage = resCompOld.extraDamage;
            if(resCompOld.lastHit.equals(target.getUUID())){
                newDamage += 0.05f;
            } else if(resCompOld.entityType.equals(targetType)){
                newDamage += 0.5f;
            } else {
                newDamage = 0.5f;
            }
            newDamage = Math.round(newDamage * 1000) / 1000f;
            return new ResonatingComponent(targetType, target.getUUID(), newDamage);
        }).orElse(new ResonatingComponent(targetType, target.getUUID(), 0.5f));
        stack.set(SwordsModComponents.RESONANCE_DATA, resComp);
        if(attacker instanceof ServerPlayer sPlayer) resComp.getDamageStateText().ifPresent(t -> sPlayer.displayClientMessage(t, true));
        super.postHurtEnemy(stack, target, attacker);
    }

    public record ResonatingComponent(ResourceLocation entityType, UUID lastHit, float extraDamage){
        public static final Codec<ResonatingComponent> CODEC = RecordCodecBuilder.create(instance ->
            instance.group(
                ResourceLocation.CODEC.fieldOf("entityType").forGetter(ResonatingComponent::entityType),
                UUIDUtil.AUTHLIB_CODEC.fieldOf("lastHit").forGetter(ResonatingComponent::lastHit),
                Codec.FLOAT.fieldOf("extraDamage").forGetter(ResonatingComponent::extraDamage)
            ).apply(instance, ResonatingComponent::new)
        );

        public static final StreamCodec<? super RegistryFriendlyByteBuf, ResonatingComponent> PACKET_CODEC = StreamCodec.composite(
            ResourceLocation.STREAM_CODEC, ResonatingComponent::entityType,
            UUIDUtil.STREAM_CODEC, ResonatingComponent::lastHit,
            ByteBufCodecs.FLOAT, ResonatingComponent::extraDamage,
            ResonatingComponent::new
        );

        public Optional<Component> getDamageStateText(){
            return BuiltInRegistries.ENTITY_TYPE.getOptional(entityType).map(
                entType -> entType.getDescription().copy().append(": " + (extraDamage == 0 ? "" : "+") + extraDamage)
                    .append(" ").append(Component.translatable("attribute.name.generic.attack_damage"))
                    .withStyle(ChatFormatting.DARK_GREEN, ChatFormatting.ITALIC));
        }
    }
}
