package com.samsthenerd.monthofswords.render;

import com.mojang.blaze3d.platform.NativeImage;
import com.samsthenerd.monthofswords.mixins.MixinAccessResTexLocation;
import net.minecraft.client.Minecraft;
import net.minecraft.client.renderer.texture.AbstractTexture;
import net.minecraft.client.renderer.texture.DynamicTexture;
import net.minecraft.client.renderer.texture.HttpTexture;
import net.minecraft.client.renderer.texture.SimpleTexture;
import net.minecraft.client.texture.*;
import net.minecraft.resources.ResourceLocation;
import net.minecraft.server.packs.resources.ResourceManager;
import net.minecraft.util.FastColor.ARGB32;
import java.io.IOException;
import java.util.HashMap;
import java.util.Optional;

public class GhostifyTexture {
    private static final HashMap<ResourceLocation, ResourceLocation> GHOSTIFIED_TEXTURES = new HashMap<>();

    public static void clearTextures(ResourceManager resMan){
        for(ResourceLocation tex : GHOSTIFIED_TEXTURES.values()){
            Minecraft.getInstance().getTextureManager().release(tex);
        }
        GHOSTIFIED_TEXTURES.clear();
    }

    public static Optional<ResourceLocation> getGhostifiedTexture(ResourceLocation originalTextureId){
        if(GHOSTIFIED_TEXTURES.containsKey(originalTextureId)) return Optional.of(GHOSTIFIED_TEXTURES.get(originalTextureId));
        AbstractTexture tex = Minecraft.getInstance().getTextureManager().getTexture(originalTextureId);
        return getImageFromTexture(tex).map(img -> {
            var newImg = new NativeImage(img.getWidth(), img.getHeight(), true);
            int minColor = 255;
            int maxColor = 0;
            for(int x = 0; x < img.getWidth(); x++){
                for(int y = 0; y < img.getHeight(); y++){
                    int color = img.getPixelRGBA(x,y);
                    int alpha = ARGB32.alpha(color);
                    if(alpha == 0){
                        newImg.setPixelRGBA(x,y,0);
                        continue;
                    }
                    int luminance = (int)((0.2126 * ARGB32.red(color)
                        + 0.7152 * ARGB32.green(color)
                        + 0.0722 * ARGB32.blue(color)
                    )); // Standard relative luminance calculation, stolen from hex :p
                    minColor = Math.min(luminance, minColor);
                    maxColor = Math.max(luminance, maxColor);
                    int argb = ARGB32.color(alpha, luminance,luminance,luminance);
                    newImg.setPixelRGBA(x,y, argb);
                }
            }
            int minColorF = minColor;
            int maxColorF = maxColor;
            double r = maxColorF - minColorF;
            newImg.applyToAllPixels(c -> {
                int b = (0x0000FF & c);
                int bn = (int)(100 * ( b - minColorF) / r) + 155;
                return ARGB32.color(ARGB32.alpha(c), bn, bn, bn);
            });
            ResourceLocation newId = ResourceLocation.fromNamespaceAndPath(originalTextureId.getNamespace() + "_monthofswords_ghostified",
                originalTextureId.getPath() + "_monthofswords_ghostified");
            Minecraft.getInstance().getTextureManager().register(newId, new DynamicTexture(newImg));
            GHOSTIFIED_TEXTURES.put(originalTextureId, newId);
            return newId;
        });
    }

    public static Optional<NativeImage> getImageFromTexture(AbstractTexture texture){
        if(texture instanceof DynamicTexture nibTex){
            return Optional.ofNullable(nibTex.getPixels());
        } else if (texture instanceof HttpTexture psTex) {
            if(psTex instanceof PlayerSkinImageDuck psiDuck){
                NativeImage img = psiDuck.mos$getSkinNativeImage();
                if(img != null) return Optional.of(img);
            }
        }
        if(texture instanceof SimpleTexture resTex){
            return ResLoaderAccessor.loadNativeImage(Minecraft.getInstance().getResourceManager(), resTex);
        }
        return Optional.empty();
    }

    public static class ResLoaderAccessor extends SimpleTexture {

        public ResLoaderAccessor(ResourceLocation location) {
            super(location);
        }

        public static Optional<NativeImage> loadNativeImage(ResourceManager resMan, SimpleTexture resTex){
            try{
                var texData = SimpleTexture.TextureImage.load(resMan, ((MixinAccessResTexLocation) resTex).getLocation());
                return Optional.of(texData.getImage());
            } catch (IOException e){
                return Optional.empty();
            }
        }
    }
}
