package com.samsthenerd.monthofswords.items;

import com.samsthenerd.monthofswords.SwordsMod;
import com.samsthenerd.monthofswords.utils.BFSHelper;
import net.minecraft.core.BlockPos;
import net.minecraft.core.Direction;
import net.minecraft.item.*;
import net.minecraft.network.chat.Style;
import net.minecraft.sounds.SoundEvents;
import net.minecraft.sounds.SoundSource;
import net.minecraft.tags.BlockTags;
import net.minecraft.util.RandomSource;
import net.minecraft.world.InteractionHand;
import net.minecraft.world.InteractionResultHolder;
import net.minecraft.world.damagesource.DamageSource;
import net.minecraft.world.effect.MobEffectInstance;
import net.minecraft.world.effect.MobEffects;
import net.minecraft.world.entity.Entity;
import net.minecraft.world.entity.EquipmentSlot;
import net.minecraft.world.entity.LivingEntity;
import net.minecraft.world.entity.player.Player;
import net.minecraft.world.entity.projectile.SmallFireball;
import net.minecraft.world.item.Item;
import net.minecraft.world.item.ItemStack;
import net.minecraft.world.item.Items;
import net.minecraft.world.item.SwordItem;
import net.minecraft.world.item.Tier;
import net.minecraft.world.item.crafting.Ingredient;
import net.minecraft.world.level.Level;
import net.minecraft.world.level.block.BaseFireBlock;
import net.minecraft.world.level.block.Blocks;
import net.minecraft.world.level.block.state.BlockState;
import net.minecraft.world.phys.AABB;
import net.minecraft.world.phys.BlockHitResult;
import net.minecraft.world.phys.Vec3;
import java.util.List;
import java.util.Map;
import java.util.function.UnaryOperator;

public class FlameSwordItem extends SwordtemberItem implements SwordActionHaverServer{

    public static final Tier FLAME_MATERIAL = new ClassyToolMaterial(1500, 7f, 3f,
            BlockTags.INCORRECT_FOR_NETHERITE_TOOL, 14, () -> Ingredient.of(Items.BLAZE_POWDER));

    public FlameSwordItem(Item.Properties itemSettings) {
        super(FLAME_MATERIAL, itemSettings.attributes(
                SwordItem.createAttributes(FLAME_MATERIAL, 3, -2.4f))
        );
    }

    @Override
    public float getAttackDamageBonus(Entity target, float baseAttackDamage, DamageSource damageSource) {
        if(damageSource.getEntity() instanceof LivingEntity attacker){
            return attacker.isOnFire() ? 1f : 0;
        }
        return 0;
    }

    @Override
    public boolean hurtEnemy(ItemStack stack, LivingEntity target, LivingEntity attacker) {
        Level world = attacker.level();
        RandomSource random = world.getRandom();
        world.playSound(null, attacker.blockPosition(), SoundEvents.FIRECHARGE_USE, SoundSource.PLAYERS, 0.5f, (random.nextFloat() - random.nextFloat()) * 0.2f + 2f);
        target.igniteForTicks(100);
        return super.hurtEnemy(stack, target, attacker);
    }

    private static SmallFireball makeSafeFireball(Player player, Vec3 vel){
        return new SmallFireball(player.level(), player, vel){
            @Override
            protected void onHitBlock(BlockHitResult bhs){
                if(SwordsMod.canBeDestructive(player, bhs.getBlockPos())){
                    super.onHitBlock(bhs);
                } else {
                    BlockState blockState = this.level().getBlockState(bhs.getBlockPos());
                    blockState.onProjectileHit(this.level(), blockState, bhs, this);
                    List<Entity> nearbyEnts = player.level().getEntities(player, new AABB(bhs.getBlockPos()).inflate(2));
                    for(Entity target : nearbyEnts){
                        target.igniteForTicks(100);
                    }
                }
            }
        };
    }

    @Override
    public InteractionResultHolder<ItemStack> use(Level world, Player player, InteractionHand hand) {
        ItemStack stack = player.getItemInHand(hand);
        if(player.getCooldowns().isOnCooldown(this)) return InteractionResultHolder.pass(stack);

        SmallFireball fireball = makeSafeFireball(player, player.getLookAngle().scale(3));
        Vec3 lookVec = player.getLookAngle();
        fireball.setPos(player.position().add(lookVec.x, 1.4, lookVec.z));
        player.level().addFreshEntity(fireball);
        player.getCooldowns().addCooldown(this, 50);
        stack.hurtAndBreak(1, player, hand == InteractionHand.MAIN_HAND ? EquipmentSlot.MAINHAND : EquipmentSlot.OFFHAND);
        RandomSource random = world.getRandom();
        world.playSound(null, player.blockPosition(), SoundEvents.FIRECHARGE_USE, SoundSource.BLOCKS, 1.0f, (random.nextFloat() - random.nextFloat()) * 0.2f + 1.0f);
        return InteractionResultHolder.success(stack);

    }

    @Override
    public boolean doSwordAction(Player player, ItemStack swordStack) {

        if (player.getCooldowns().isOnCooldown(this)) return false;

        Level world = player.level();
        Vec3 lookVec = player.getLookAngle();

        Vec3 sideVec = lookVec.cross(new Vec3(0, 1, 0)).normalize().scale(0.1f);
        Vec3[] sideVels = {sideVec, new Vec3(0, 0, 0), sideVec.scale(-1)};
        Vec3[] verVels = {new Vec3(0, 0.1f, 0), new Vec3(0, 0, 0), new Vec3(0, -0.1f, 0)};

        for (Vec3 vVel : verVels) {
            for (Vec3 hVel : sideVels) {
                SmallFireball fireballCenter = makeSafeFireball(player,
                    player.getLookAngle().add(vVel).add(hVel).scale(3));
                fireballCenter.setPos(player.position().add(lookVec.x, 1.4, lookVec.z));
                player.level().addFreshEntity(fireballCenter);
            }
        }

        player.forceAddEffect(new MobEffectInstance(MobEffects.FIRE_RESISTANCE, 400), player);

        if (SwordsMod.canBeDestructive(player, null)) {
            Map<BlockPos, Integer> fireZone = BFSHelper.runBFS(world, player.blockPosition(),
                (worldArg, pos, dist) -> true, 3, false);

            for (BlockPos fPos : fireZone.keySet()) {
                if (world.getBlockState(fPos).getBlock() == Blocks.AIR
                    && world.getBlockState(fPos.relative(Direction.DOWN)).getBlock() != Blocks.AIR) {
                    world.setBlockAndUpdate(fPos, BaseFireBlock.getState(world, fPos));
                }
            }
        } else {
            List<Entity> nearbyEnts = world.getEntities(player, new AABB(player.blockPosition()).inflate(3));
            for(Entity target : nearbyEnts){
                target.igniteForTicks(100);
            }
        }

        RandomSource random = world.getRandom();
        player.getCooldowns().addCooldown(this, 200);
        world.playSound(null, player.blockPosition(), SoundEvents.BLAZE_SHOOT, SoundSource.PLAYERS, 1.0f, (random.nextFloat() - random.nextFloat()) * 0.2f + 1.0f);
        swordStack.hurtAndBreak(15, player, player.getMainHandItem() == swordStack ? EquipmentSlot.MAINHAND : EquipmentSlot.OFFHAND);
        return true;
    }

    @Override
    public UnaryOperator<Style> getSwordTooltipStyleModifier(){
        return (style) -> style.withColor(0xfc6f03);
    }
}
