package com.samsthenerd.inline.impl;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.util.concurrent.AtomicDouble;
import com.mojang.blaze3d.pipeline.TextureTarget;
import com.mojang.blaze3d.platform.NativeImage;
import com.mojang.blaze3d.systems.RenderSystem;
import com.mojang.blaze3d.vertex.PoseStack;
import com.mojang.blaze3d.vertex.Tesselator;
import com.mojang.blaze3d.vertex.VertexSorting;
import com.samsthenerd.inline.Inline;
import com.samsthenerd.inline.api.InlineData;
import com.samsthenerd.inline.api.client.GlowHandling;
import com.samsthenerd.inline.api.client.InlineClientAPI;
import com.samsthenerd.inline.api.client.InlineRenderer;
import com.samsthenerd.inline.mixin.core.MixinSetTessBuffer;
import com.samsthenerd.inline.utils.*;
import it.unimi.dsi.fastutil.ints.Int2IntMap;
import it.unimi.dsi.fastutil.ints.Int2IntOpenHashMap;
import org.joml.Matrix3f;
import org.joml.Matrix4f;
import org.joml.Matrix4fStack;

import java.time.Duration;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.Queue;
import java.util.Set;
import net.minecraft.client.Minecraft;
import net.minecraft.client.gui.Font;
import net.minecraft.client.gui.GuiGraphics;
import net.minecraft.client.renderer.MultiBufferSource;
import net.minecraft.client.renderer.texture.DynamicTexture;
import net.minecraft.network.chat.Style;
import net.minecraft.resources.ResourceLocation;
import net.minecraft.util.FastColor;
import net.minecraft.util.Tuple;

public class InlineRenderCore {

    private static TextureTarget GLOW_BUFF = new TextureTarget(128, 128, true, Minecraft.ON_OSX);

    private static Set<ResourceLocation> ERRORED_RENDERERS = new HashSet<>();

    public static boolean textDrawerAcceptHandler(int index, Style style, int codepoint, RenderArgs args) {
        InlineData inlData = style.getInlineData();
        if(inlData == null) return false;
        InlineRenderer renderer = InlineClientAPI.INSTANCE.getRenderer(inlData.getRendererId());
        if(renderer == null) return false;

        boolean noGlowHandle = !(renderer.getGlowPreference(inlData) instanceof GlowHandling.None);
        boolean hasGlowyMarker = style.getComponent(InlineStyle.GLOWY_MARKER_COMP);
        if(noGlowHandle && hasGlowyMarker){
            return true;
        }

        int glowColor = style.getComponent(InlineStyle.GLOWY_PARENT_COMP);
        boolean needsGlowChildren = glowColor != -1 && renderer.getGlowPreference(inlData) instanceof GlowHandling.Full;

        Tesselator heldTess = Tesselator.getInstance();
        MixinSetTessBuffer.setInstance(secondaryTess);

        MultiBufferSource.BufferSource immToUse = args.provider() instanceof MultiBufferSource.BufferSource imm
                ? imm : VCPImmediateButImLyingAboutIt.of(args.provider());
        immToUse.endBatch();

        GuiGraphics drawContext = new GuiGraphics(Minecraft.getInstance(), immToUse);
        PoseStack matrices = drawContext.pose();
        matrices.pushPose();

        double sizeMod = style.getComponent(InlineStyle.SIZE_MODIFIER_COMP);

        // note that we need to deal with position before normals now bc of matrixstack change.
        matrices.mulPose(args.matrix());
        matrices.mulPose(new Matrix4f().scale(1f, 1f, 0.001f));
        // there's almost certainly a better way to do this, but we're just flipping the y and z axes
        matrices.last().normal().mul(new Matrix3f(1, 0, 0, 0, 0, 1, 0, 1, 0));
        // then inverting. idk why this is necessary.
        matrices.last().normal().scale(-1);

        matrices.translate(args.x(), args.y(), 0);

        // only handle sizing here if sizing exists, renderer won't handle it, and player config says it's ok
        double maxSizeMod = InlineClientAPI.INSTANCE.getConfig().maxChatSizeModifier();
        if(sizeMod > maxSizeMod && com.samsthenerd.inline.api.client.InlineRenderer.isFlat(matrices, args.layerType) && com.samsthenerd.inline.api.client.InlineRenderer.isChatty()) sizeMod = maxSizeMod;

        boolean needToHandleSize = sizeMod != 1.0 && !renderer.handleOwnSizing(inlData);

        if(needToHandleSize){
            double yOffset = (sizeMod - 1) * 4; // sizeMod - 1 gives how much goes "outside" the main 8px. scale by 8 and then take half, so x4.
            matrices.translate(0, -yOffset, 0);
            matrices.scale((float)sizeMod, (float)sizeMod, 1f);
        }

        float alphaToUse = (args.alpha() == 0 ? 1 : args.alpha());

        int rendererARGB = FastColor.ARGB32.color(
                Math.round(alphaToUse* 255), Math.round(args.red() * 255),
                Math.round(args.green() * 255), Math.round(args.blue() * 255)
        );
        int usableColor = rendererARGB;
        if(style.getColor() != null){
            usableColor = FastColor.ARGB32.multiply(rendererARGB, style.getColor().getValue() | 0xFF_000000);
        }

        InlineRenderer.TextRenderingContext trContext = new InlineRenderer.TextRenderingContext(args.light(), args.shadow(), args.brightnessMultiplier(),
                args.red(), args.green(), args.blue(), args.alpha() == 0 ? 1 : args.alpha(), args.layerType(), args.provider(), style.getComponent(InlineStyle.GLOWY_MARKER_COMP),
                style.getComponent(InlineStyle.GLOWY_PARENT_COMP), usableColor);

        if(!renderer.handleOwnTransparency(inlData)){
            RenderSystem.setShaderColor(1, 1, 1, alphaToUse);
        }

        try {
            if(needsGlowChildren){
                Tuple<Spritelike, Runnable> texResult = getGlowTextureSprite(inlData, renderer, immToUse, sizeMod, index, style, codepoint, trContext);
                Spritelike backSprite = texResult.getA();
                int brighterGlow = ColorUtils.ARGBtoHSB(glowColor)[2] > ColorUtils.ARGBtoHSB(usableColor)[2] ? glowColor : usableColor;
                SpritelikeRenderers.getRenderer(backSprite).drawSpriteWithLight(backSprite, drawContext, -2, -4, 0, 16, 16, trContext.light(), brighterGlow);
                texResult.getB().run(); // cleanup if needed
            }
            matrices.translate(0, 0, 10);
                args.xUpdater().addAndGet(renderer.render(inlData, drawContext, index, style, codepoint, trContext) * (needToHandleSize ? (float)sizeMod : 1f));

            immToUse.endBatch();
        } catch(Exception e){
            if(ERRORED_RENDERERS.add(inlData.getRendererId())){
                Inline.LOGGER.error("Error rendering inline with renderer: " + inlData.getRendererId().toString()
                    + " -- To prevent logspam this will only show once. This is likely an issue with whatever mod adds the renderer.");
            }
        }


        if(!renderer.handleOwnTransparency(inlData)){
            RenderSystem.setShaderColor(1f, 1f, 1f, 1f);
        }

        matrices.popPose();

        MixinSetTessBuffer.setInstance(heldTess);

        return true;
    }

    private static final Cache<String, Spritelike> GLOW_TEXTURE_CACHE = CacheBuilder.newBuilder()
            .maximumSize(100) // should be a decent size ?
            .expireAfterAccess(Duration.ofMinutes(5))
            .removalListener((notif) -> {
                if(notif.getValue() instanceof TextureSprite tSprite){
                    Minecraft.getInstance().getTextureManager().release(tSprite.getTextureId());
                }
            })
            .build();

    private static Tuple<Spritelike, Runnable> getGlowTextureSprite(InlineData inlData, InlineRenderer renderer, MultiBufferSource.BufferSource immToUse,
                                                                   double sizeMod, int index, Style style, int codepoint, InlineRenderer.TextRenderingContext trContext){

        double outlineScaleBack = 1 / sizeMod;
        boolean needToHandleSize = sizeMod != 1.0 && !renderer.handleOwnSizing(inlData);

        // this should never happen, i'm just casting
        if(!(renderer.getGlowPreference(inlData) instanceof GlowHandling.Full fullHandling)) return null;

        String texCacheId = null;
        if(fullHandling.cacheId != null){
            texCacheId = "inlineglowtexture" + renderer.getId().toLanguageKey() + "." + fullHandling.cacheId + sizeMod; // should sizemod even be here ?
            Spritelike texSpriteMaybe = GLOW_TEXTURE_CACHE.getIfPresent(texCacheId);
            if(texSpriteMaybe != null){
                return new Tuple<>(texSpriteMaybe, () -> {});
            }
        }

        int resScale = 8;
        GLOW_BUFF.setClearColor(0, 0, 0, 0);
        GLOW_BUFF.clear(false);
        Minecraft.getInstance().getMainRenderTarget().unbindWrite();
        Matrix4fStack mvStack = RenderSystem.getModelViewStack();
        Matrix4f backupProjMatrix = RenderSystem.getProjectionMatrix();
        mvStack.pushMatrix();
        mvStack.identity();
        RenderSystem.applyModelViewMatrix();
        VertexSorting backupVertexSorter = RenderSystem.getVertexSorting();
        RenderSystem.backupProjectionMatrix();
        Matrix4f newProjMatrix = new Matrix4f();
        newProjMatrix.identity();
        newProjMatrix.setOrtho(0, 16*resScale, 0,16*resScale, 0, 100);
        RenderSystem.setProjectionMatrix(newProjMatrix, VertexSorting.DISTANCE_TO_ORIGIN);
        GLOW_BUFF.bindWrite(true);
        GuiGraphics glowContext = new GuiGraphics(Minecraft.getInstance(), immToUse);
        PoseStack glowStack = glowContext.pose();

        glowStack.pushPose();
        glowStack.translate(2*resScale, 4 * resScale, -50);
        glowStack.scale(resScale, resScale, 1f);
        glowStack.mulPose(new Matrix4f().scale(1, 1, 0.01f));
        float xOffsetDiff = renderer.render(inlData, glowContext, index, style, codepoint, trContext) * (needToHandleSize ? (float)sizeMod : 1f);
//            args.xUpdater().addAndGet(xOffsetDiff);

        immToUse.endBatch();

        mvStack.popMatrix();
        RenderSystem.applyModelViewMatrix();
        RenderSystem.setProjectionMatrix(backupProjMatrix, backupVertexSorter);
        GLOW_BUFF.unbindWrite();

        try (NativeImage nativeImage = new NativeImage(16*resScale, 16*resScale, true)) {
            GLOW_BUFF.bindRead();
            nativeImage.downloadTexture(0, false);
            GLOW_BUFF.unbindRead();
//                nativeImage.writeTo(MinecraftClient.getInstance().runDirectory.toPath().resolve("boop.png"));
//                nativeImage.apply(original -> ColorHelper.Argb.getAlpha(original) << 24 | 0x00_FFFFFF);
            NativeImage fullImage = new NativeImage(nativeImage.getWidth(), nativeImage.getHeight(), true);
            int outlineRange = (int)Math.round(resScale * outlineScaleBack);
            // would
            Queue<Integer> pixelQueue = new LinkedList<>(); // bfs guarantees that we only hit each one once
            Int2IntMap seenPixels = new Int2IntOpenHashMap(); // x + y * width -> distance from an original
            // do an initial sweep for starting ones
            int imgWidth = nativeImage.getWidth();
            int imgHeight = nativeImage.getHeight();
            for(int px = 0; px < imgWidth; px++){
                for(int py = 0; py < imgHeight; py++){
                    if(FastColor.ARGB32.alpha(nativeImage.getPixelRGBA(px, py)) == 0) continue;
                    int thisPos = px + py *imgWidth;
                    seenPixels.put(thisPos, 0);
                    pixelQueue.add(thisPos);
                }
            }
            nativeImage.close();
            while(!pixelQueue.isEmpty()){
                int cPix = pixelQueue.poll(); // get current pixel to process;
                int cX = cPix % imgWidth;
                int cY = cPix / imgWidth;
                fullImage.setPixelRGBA(cX, cY, 0xFF_FFFFFF);
                if(seenPixels.get(cPix) >= outlineRange) continue; // exit out if we don't need to add neighbors
                for(int i = -1; i <= 1; i++){
                    if( cX + i < 0 || cX + i >= imgWidth) continue;
                    for(int j = -1; j <= 1; j++){
                        if( cY + j < 0 || cY + j >= imgHeight) continue;
                        int nbrPos = cPix + i + j * imgWidth;
                        if(!seenPixels.containsKey(nbrPos)){
                            seenPixels.put(nbrPos, seenPixels.get(cPix)+1);
                            pixelQueue.add(nbrPos);
                        }
                    }
                }
            }

            Minecraft.getInstance().getMainRenderTarget().bindWrite(true);
            if(texCacheId != null){
                ResourceLocation backTexId = Minecraft.getInstance().getTextureManager().register(
                        texCacheId,
                        new DynamicTexture(fullImage));
                TextureSprite tSprite = new TextureSprite(backTexId);
                GLOW_TEXTURE_CACHE.put(texCacheId, tSprite);
                return new Tuple<>(tSprite, () -> {});
            } else {
                ResourceLocation backTexId = Minecraft.getInstance().getTextureManager().register(Inline.id("glowtextureback").toLanguageKey(), new DynamicTexture(fullImage));
                return new Tuple<>(new TextureSprite(backTexId), () -> {
                    Minecraft.getInstance().getTextureManager().release(backTexId);
                });
            }
        } catch (Exception e){
            Inline.LOGGER.error(e.toString());
        }
        return null;
    }

    private static final Tesselator secondaryTess = new Tesselator();

    public record RenderArgs(float x, float y, Matrix4f matrix, int light, boolean shadow, float brightnessMultiplier,
                                    float red, float green, float blue, float alpha, Font.DisplayMode layerType,
                                    MultiBufferSource provider, AtomicDouble xUpdater){}
}
