package com.samsthenerd.inline.api.matching;

import java.util.Optional;
import java.util.concurrent.atomic.AtomicInteger;
import net.minecraft.network.chat.Component;
import net.minecraft.network.chat.Style;
import net.minecraft.util.FormattedCharSink;
import com.samsthenerd.inline.api.InlineData;

/**
 * Represents some match from a matcher. 
 * <p>
 * A match is attached to some input text by a matcher and is expected to 
 * provide styled characters to the CharacterVisitor in place of the text segment 
 * it's attached to.
 * <p>
 * {@link DataMatch} will work for most use cases.
 */
public interface InlineMatch {
    /**
     * Supplies styled characters
     * @param visitor accepts the styled characters
     * @param index the number of characters given to this visitor so far
     * @param currentStyle the style that this text would otherwise have.
     * @return the number of characters supplied to the visitor.
     */
    public int accept(FormattedCharSink visitor, int index, Style currentStyle);

    /**
     * Gets the length of this match
     * @return the number of characters we plan to give the visitor.
     */
    public int charLength();

    /**
     * A match representing some data.
     */
    public static class DataMatch implements InlineMatch{

        public final InlineData data;
        public final Style style;

        /**
         * Makes a DataMatch with the given data and style
         * @param data
         * @param style
         */
        public DataMatch(InlineData data, Style style){
            this.data = data;
            this.style = style;
        }

        /**
         * Makes a DataMatch with the given data
         * @param data
         */
        public DataMatch(InlineData data){
            this(data, Style.f_131099_);
        }

        public int accept(FormattedCharSink visitor, int index, Style currentStyle){
            Style nonDataStyle = style.m_131146_(currentStyle);
            Style dataStyle = nonDataStyle.withInlineData(data);
            visitor.m_6411_(index, dataStyle, '.');
            return 1;
        }

        public int charLength(){
            return 1;
        }
    }

    /**
     * A match representing some text.
     */
    public static class TextMatch implements InlineMatch{

        private Component text;

        /**
         * Makes a TextMatch with the given text
         * @param text
         */
        public TextMatch(Component text){
            this.text = text;
        }

        public int accept(FormattedCharSink visitor, int index, Style currentStyle){
            if(text.getString().equals("")) return 1;
            AtomicInteger offset = new AtomicInteger(0);
            text.m_7451_((Style style, String str)->{
                for(char c : str.toCharArray()){
                    visitor.m_6411_(index + offset.get(), style.m_131146_(currentStyle), c);
                    offset.incrementAndGet();
                }
                return Optional.empty();
            }, currentStyle);
            return offset.get();
        }

        public int charLength(){
            return Math.max(text.getString().length(), 1);
        }
    }
}
