package com.samsthenerd.inline.tooltips;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import javax.annotation.Nullable;

import org.jetbrains.annotations.NotNull;

import com.mojang.serialization.Codec;
import com.mojang.serialization.DataResult;
import com.samsthenerd.inline.Inline;
import net.minecraft.client.gui.screens.inventory.tooltip.ClientTooltipComponent;
import net.minecraft.nbt.CompoundTag;
import net.minecraft.nbt.NbtOps;
import net.minecraft.network.chat.Component;
import net.minecraft.network.chat.HoverEvent;
import net.minecraft.resources.ResourceLocation;
import net.minecraft.world.inventory.tooltip.TooltipComponent;
import net.minecraft.world.item.Item;
import net.minecraft.world.item.ItemStack;
import net.minecraft.world.item.Items;

/**
 * An item-less custom tooltip system.
 * <p>
 * By default Minecraft only supports 3 types of {@link HoverEvent}s:
 * Text, Item, and Entity (which is kinda just text again). But, Item tooltips
 * have a pretty good tooltip system with {@link TooltipComponent} and {@link ClientTooltipComponent}s.
 * Unfortunately, using those requires adding a new item, which is rather inconvenient, so
 * instead we hijack an arbitrary vanilla item to throw our tooltip system on. 
 * <p>
 * You can use these tooltips in hover events by making an item based hover event
 * out of the dummy itemstack from {@link CustomTooltipManager#getForTooltip(CustomTooltipProvider, Object)}.
 * <p>
 * Note that this isn't necessarily a part of the core Inline API, just a useful system
 * used for the built in player facing features.
 * @see CustomTooltipProvider
 */
public class CustomTooltipManager {

    public static final Item HIJACKED_ITEM = Items.f_42402_;

    /**
     * Gets a dummy itemstack hijacked with the data to display the tooltip 
     * provided by the provider for the given content. 
     * @param <T> Type of the content that the provider handles
     * @param provider A tooltip provider that can handle the given content.
     * @param content Arbitrary data to be handled by the provider.
     * @return an itemstack that will have a tooltip given by our provider and content.
     */
    public static <T> ItemStack getForTooltip(CustomTooltipProvider<T> provider, T content){
        ItemStack stack = new ItemStack(HIJACKED_ITEM);
        CompoundTag tag = new CompoundTag();
        tag.m_128359_("id", provider.getId().toString());
        tag.m_128365_("data", provider.getCodec().encodeStart(NbtOps.f_128958_, content).getOrThrow(false, Inline.LOGGER::error));
        stack.m_41700_("inlinecustomtooltip", tag);
        return stack;
    }

    @Nullable
    public static CustomTooltipProvider<?> getProvider(ItemStack stack){
        CompoundTag tag = stack.m_41737_("inlinecustomtooltip");
        // we're throwing this all in a try/catch because it's good enough 
        try{
            return PROVIDERS.get(new ResourceLocation(tag.m_128461_("id")));
        } catch(Exception e) {
            return null;
        }
    }

    @Nullable
    public static <T> List<Component> getTooltipText(ItemStack stack){
        try{
            CompoundTag tag = stack.m_41737_("inlinecustomtooltip");
            CustomTooltipProvider<T> provider = (CustomTooltipProvider<T>)getProvider(stack);
            DataResult<T> contentRes = provider.getCodec().parse(NbtOps.f_128958_, tag.m_128423_("data"));
            T content = contentRes.resultOrPartial(Inline.LOGGER::error).orElseThrow();
            return provider.getTooltipText(content);
        } catch(Exception e) {
            return null;
        }
    }

    @Nullable
    public static <T> Optional<TooltipComponent> getTooltipData(ItemStack stack){
        try{
            CompoundTag tag = stack.m_41737_("inlinecustomtooltip");
            CustomTooltipProvider<T> provider = (CustomTooltipProvider<T>)getProvider(stack);
            DataResult<T> contentRes = provider.getCodec().parse(NbtOps.f_128958_, tag.m_128423_("data"));
            T content = contentRes.resultOrPartial(Inline.LOGGER::error).orElseThrow();
            return provider.getTooltipData(content);
        } catch(Exception e) {
            return null;
        }
    }

    private static final Map<ResourceLocation, CustomTooltipProvider> PROVIDERS = new HashMap<>();

    /**
     * Registers a provider.
     * @param <T> type that the provider handles
     * @param provider
     * @return the same provider.
     */
    public static <T> CustomTooltipProvider<T> registerProvider(CustomTooltipProvider<T> provider){
        PROVIDERS.put(provider.getId(), provider);
        return provider;
    }

    /**
     * Makes a tooltip out of some arbitrary data of type T.
     * <p>
     * Delegating between providers is handled by the manager.
     */
    public static interface CustomTooltipProvider<T>{

        public ResourceLocation getId();

        @NotNull
        public List<Component> getTooltipText(T content);

        @NotNull
        public Optional<TooltipComponent> getTooltipData(T content);

        @NotNull
        public Codec<T> getCodec();
    }
}
