package com.samsthenerd.inline.utils.cradles;

import com.mojang.authlib.GameProfile;
import com.mojang.datafixers.util.Either;
import com.mojang.serialization.Codec;
import com.samsthenerd.inline.Inline;
import com.samsthenerd.inline.utils.EntityCradle;
import com.samsthenerd.inline.utils.FakeClientPlayerMaker;
import java.util.HashMap;
import java.util.UUID;
import net.minecraft.core.UUIDUtil;
import net.minecraft.resources.ResourceLocation;
import net.minecraft.util.Tuple;
import net.minecraft.world.entity.Entity;
import net.minecraft.world.level.Level;

/**
 * An entity cradle backed by a player GameProfile
 */
public class PlayerCradle extends EntityCradle {

    private static final HashMap<UUID, Entity> UUID_PLAYER_CACHE = new HashMap<>();
    private static final HashMap<String, Entity> NAME_PLAYER_CACHE = new HashMap<>();

    public static final Codec<GameProfile> GAME_PROFILE_CODEC = Codec.either(
            Codec.STRING.fieldOf("username").codec(), 
            UUIDUtil.f_252480_.fieldOf("uuid").codec()
        ).xmap(
            (nameOrId) -> nameOrId.map(
                name -> new GameProfile(null, name), 
                uuid -> new GameProfile(uuid, null)),
            
            (profile) -> {
                String name = profile.getName();
                if(name != null && !name.isEmpty()){
                    return Either.left(name);
                }
                return Either.right(profile.getId());
            }
        );

    private GameProfile profile;

    public PlayerCradle(GameProfile profile){
        this.profile = profile;
    }

    public GameProfile getProfile(){
        return profile;
    }

    public CradleType<?> getType(){
        return PlayerCradleType.INSTANCE;
    }

    @Override
    public String getId(){
        return profile.getId() == null ? profile.getName() : profile.getId().toString();
    }

    public Entity getEntity(Level world){
        UUID playerId = profile.getId();
        if(playerId != null && UUID_PLAYER_CACHE.containsKey(playerId)){
            return UUID_PLAYER_CACHE.get(playerId);
        }
        String playerName = profile.getName();
        if(playerName != null && !playerName.equals("") && NAME_PLAYER_CACHE.containsKey(playerName)){
            return NAME_PLAYER_CACHE.get(playerName);
        }

        if(!world.m_5776_()){
            return null;
        }
        
        Tuple<Entity, Boolean> playerRes = FakeClientPlayerMaker.getPlayerEntity(profile);
        if(playerRes.m_14419_() && playerId != null){
            UUID_PLAYER_CACHE.put(playerId, playerRes.m_14418_());
        }
        if(playerRes.m_14419_() && playerName != null && !playerName.equals("")){
            NAME_PLAYER_CACHE.put(playerName, playerRes.m_14418_());
        }
        return playerRes.m_14418_();
    }

    private static class PlayerCradleType implements CradleType<PlayerCradle>{

        public static PlayerCradleType INSTANCE = EntityCradle.addCradleType(new PlayerCradleType());

        public ResourceLocation getId(){
            return new ResourceLocation(Inline.MOD_ID, "nbt");
        }

        public Codec<PlayerCradle> getCodec(){
            return GAME_PROFILE_CODEC.xmap(
                PlayerCradle::new,
                PlayerCradle::getProfile
            );
        }
    }
}
