package com.samsthenerd.inline.utils;

import com.mojang.datafixers.util.Either;
import com.samsthenerd.inline.Inline;
import com.samsthenerd.inline.utils.ImgFormatParser.ImgParseResult;
import com.samsthenerd.inline.utils.URLTextureUtils.LoadingState.StrState;
import com.samsthenerd.inline.utils.URLTextureUtils.LoadingState.SuccessState;
import javax.annotation.Nullable;
import net.minecraft.class_1047;
import net.minecraft.class_2960;
import net.minecraft.class_310;
import net.minecraft.class_3545;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.net.URL;
import java.net.URLConnection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.CompletableFuture;

public class URLTextureUtils {

    private static final Map<class_2960, LoadingState> LOADING_TEXTURE_STATES = Collections.synchronizedMap(new HashMap<>());

    public sealed interface LoadingState permits StrState, SuccessState{
        record StrState(String st) implements LoadingState{}
        LoadingState ERROR = new StrState("error");
        LoadingState IN_PROGRESS = new StrState("progress");
        record SuccessState(IntPair dims, SpriteUVLens lens, class_2960 textId) implements  LoadingState{};
    }

    // informed by hellozyemlya on discord
    @Nullable
    public static class_2960 loadTextureFromURL(String urlStr, class_2960 textureId){
        LoadingState state = LOADING_TEXTURE_STATES.get(textureId);
        if(state == LoadingState.IN_PROGRESS) return null;
        if(state == LoadingState.ERROR) return class_1047.method_4539();
        if(state instanceof SuccessState succSt){
            return succSt.textId();
        }
        if(state != null) return class_1047.method_4539(); // this shouldn't happen but just to be safe
        LOADING_TEXTURE_STATES.put(textureId, LoadingState.IN_PROGRESS);
        CompletableFuture.runAsync(() -> {
            try {
                URL textureUrl = URI.create(urlStr).toURL();
                var conn = textureUrl.openConnection();
                InputStream stream = conn.getInputStream();

                String contentType = URLConnection.guessContentTypeFromStream(stream);
                if (contentType == null) contentType = conn.getContentType();

                ImgFormatParser parser = ImgFormatParser.getFormatParser(contentType);

                Either<ImgParseResult, String> imgResult = parser.tryParse(stream, contentType);

                if (imgResult.left().isPresent()) { // success
                    ImgParseResult res = imgResult.left().get();
                    class_310.method_1551().execute(() -> {
                        LOADING_TEXTURE_STATES.put(textureId,
                            new SuccessState(res.dims(), res.lens(), res.registerTextureCallback().apply(textureId))
                        );
                    });
                } else { // error
                    String err = imgResult.right().get();
                    Inline.LOGGER.error("Failed to parse image " + urlStr + " : " + err);
                    LOADING_TEXTURE_STATES.put(textureId, LoadingState.ERROR);
                }
            } catch (IOException e){
                Inline.LOGGER.error("Failed to parse image " + urlStr + " : " + e);
                LOADING_TEXTURE_STATES.put(textureId, LoadingState.ERROR);
            }
        });
        return null;
    }


    @Nullable
    public static class_3545<IntPair, SpriteUVLens> getTextureInfo(class_2960 textureId){
        LoadingState state = LOADING_TEXTURE_STATES.get(textureId);
        if(state == LoadingState.IN_PROGRESS) return null;
        if(state == LoadingState.ERROR) return ERROR_TEXT_INFO;
        if(state instanceof SuccessState succSt){
            return new class_3545<>(succSt.dims, succSt.lens());
        }
        return null;
    }

    private static final class_3545<IntPair, SpriteUVLens> ERROR_TEXT_INFO = new class_3545<>(new IntPair(16,16), SpriteUVRegion.FULL.asLens());

    static {
        // static inits not allowed in interface.
        ImgFormatParser.loadBuiltinParsers();
    }
}
