package com.samsthenerd.inline.api;

import com.mojang.serialization.Codec;
import com.samsthenerd.inline.api.matching.InlineMatcher;
import com.samsthenerd.inline.api.client.InlineRenderer;
import com.samsthenerd.inline.impl.InlineStyle;
import net.minecraft.class_2561;
import net.minecraft.class_2583;
import net.minecraft.class_2960;

/**
 * Represents some arbitrary data that should be renderered with the
 * specified <code>InlineRenderer</code>.
 * 
 * <p>
 * For the data to be rendered, it must be attached to some Text's Style. 
 * Usually this is done by an {@link InlineMatcher} on the client so that the data
 * can be easily attached anywhere with plain text, however in some cases it may be 
 * more convenient to manually attach the data. 
 * 
 * <p>
 * Unlike most other Inline API classes, InlineData exists on both the client
 * and the server. 
 */
public interface InlineData<Self extends InlineData<Self>> {

    /**
     * Gets the InlineDataType of this data.
     * Used mostly for serialization.
     * @return type of this data.
     */
    InlineDataType<Self> getType();

    /**
     * Gets which {@link InlineRenderer} should be used for rendering this data.
     * @return id of renderer
     */
    class_2960 getRendererId();

    /**
     * Gets a suitable Style for this data, without the data itself.
     * <p>
     * This is primarily used for adding hover or click events.
     * @return Style with no attached data
     */
    default class_2583 getExtraStyle(){
        return class_2583.field_24360;
    }

    /**
     * Gets a Style with this data attached.
     * @param withExtra whether or not extra styling, such as hover or click events, should be added.
     * @return Style with attached data and extra styling if withExtra is true
     */
    default class_2583 asStyle(boolean withExtra){
        class_2583 dataStyle = InlineStyle.fromInlineData(this);
        if(withExtra){
            dataStyle = dataStyle.method_27702(getExtraStyle());
        }
        return dataStyle;
    }

    /**
     * Gets Text with this data attached.
     * <p>
     * The actual Text string is left up to the implementation.
     * @param withExtra whether or not extra styling, such as hover or click events, should be added.
     * @return Text with attached data and extra styling if withExtra is true
     */
    default class_2561 asText(boolean withExtra){
        return class_2561.method_43470(".").method_10862(asStyle(withExtra));
    }

    /**
     * The type of some InlineData.
     * Used mostly for deserialization. 
     * <p>
     * Make sure to register each type with {@link InlineAPI#addDataType(InlineDataType)}
     */
    interface InlineDataType<D extends InlineData<D>> {

        /**
         * Gets a unique identifier for this data type. 
         * Used for deserialization
         * @return id
         */
        public class_2960 getId();

        /**
         * Gets a codec for serializing data of type D.
         * @return codec
         */
        public Codec<D> getCodec();
    }
}
