package com.samsthenerd.inline.api.matching;

import java.util.Optional;
import java.util.concurrent.atomic.AtomicInteger;
import net.minecraft.class_2561;
import net.minecraft.class_2583;
import net.minecraft.class_5224;
import com.samsthenerd.inline.api.InlineData;

/**
 * Represents some match from a matcher. 
 * <p>
 * A match is attached to some input text by a matcher and is expected to 
 * provide styled characters to the CharacterVisitor in place of the text segment 
 * it's attached to.
 * <p>
 * {@link DataMatch} will work for most use cases.
 */
public interface InlineMatch {
    /**
     * Supplies styled characters
     * @param visitor accepts the styled characters
     * @param index the number of characters given to this visitor so far
     * @param currentStyle the style that this text would otherwise have.
     * @return the number of characters supplied to the visitor.
     */
    public int accept(class_5224 visitor, int index, class_2583 currentStyle);

    /**
     * Gets the length of this match
     * @return the number of characters we plan to give the visitor.
     */
    public int charLength();

    /**
     * A match representing some data.
     */
    public static class DataMatch implements InlineMatch{

        public final InlineData data;
        public final class_2583 style;

        /**
         * Makes a DataMatch with the given data and style
         * @param data
         * @param style
         */
        public DataMatch(InlineData data, class_2583 style){
            this.data = data;
            this.style = style;
        }

        /**
         * Makes a DataMatch with the given data
         * @param data
         */
        public DataMatch(InlineData data){
            this(data, class_2583.field_24360);
        }

        public int accept(class_5224 visitor, int index, class_2583 currentStyle){
            class_2583 nonDataStyle = style.method_27702(currentStyle);
            class_2583 dataStyle = nonDataStyle.withInlineData(data);
            visitor.accept(index, dataStyle, '.');
            return 1;
        }

        public int charLength(){
            return 1;
        }
    }

    /**
     * A match representing some text.
     */
    public static class TextMatch implements InlineMatch{

        private class_2561 text;

        /**
         * Makes a TextMatch with the given text
         * @param text
         */
        public TextMatch(class_2561 text){
            this.text = text;
        }

        public int accept(class_5224 visitor, int index, class_2583 currentStyle){
            if(text.getString().equals("")) return 1;
            AtomicInteger offset = new AtomicInteger(0);
            text.method_27658((class_2583 style, String str)->{
                for(char c : str.toCharArray()){
                    visitor.accept(index + offset.get(), style.method_27702(currentStyle), c);
                    offset.incrementAndGet();
                }
                return Optional.empty();
            }, currentStyle);
            return offset.get();
        }

        public int charLength(){
            return Math.max(text.getString().length(), 1);
        }
    }
}
