package com.samsthenerd.inline.tooltips;

import com.mojang.serialization.Codec;
import com.mojang.serialization.DataResult;
import com.mojang.serialization.MapCodec;
import org.jetbrains.annotations.NotNull;

import javax.annotation.Nullable;
import net.minecraft.class_1792;
import net.minecraft.class_1799;
import net.minecraft.class_1802;
import net.minecraft.class_2509;
import net.minecraft.class_2561;
import net.minecraft.class_2568;
import net.minecraft.class_2960;
import net.minecraft.class_5632;
import net.minecraft.class_5684;
import net.minecraft.class_9279;
import net.minecraft.class_9334;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

/**
 * An item-less custom tooltip system.
 * <p>
 * By default Minecraft only supports 3 types of {@link class_2568}s:
 * Text, Item, and Entity (which is kinda just text again). But, Item tooltips
 * have a pretty good tooltip system with {@link class_5632} and {@link class_5684}s.
 * Unfortunately, using those requires adding a new item, which is rather inconvenient, so
 * instead we hijack an arbitrary vanilla item to throw our tooltip system on. 
 * <p>
 * You can use these tooltips in hover events by making an item based hover event
 * out of the dummy itemstack from {@link CustomTooltipManager#getForTooltip(CustomTooltipProvider, Object)}.
 * <p>
 * Note that this isn't necessarily a part of the core Inline API, just a useful system
 * used for the built in player facing features.
 * @see CustomTooltipProvider
 */
public class CustomTooltipManager {

    public static final class_1792 HIJACKED_ITEM = class_1802.field_8153;
    private static final String TOOLTIP_DATA_KEY = "inlinecustomtooltip";

    /**
     * Gets a dummy itemstack hijacked with the data to display the tooltip 
     * provided by the provider for the given content. 
     * @param <T> Type of the content that the provider handles
     * @param provider A tooltip provider that can handle the given content.
     * @param content Arbitrary data to be handled by the provider.
     * @return an itemstack that will have a tooltip given by our provider and content.
     */
    //TODO: Test this on a dedicated server
    public static <T> class_1799 getForTooltip(CustomTooltipProvider<T> provider, T content){
        class_1799 stack = new class_1799(HIJACKED_ITEM);
        var ctpd = new CTPData<T>(provider, content);

        stack.method_57368(class_9334.field_49628, class_9279.field_49302, tagComp ->
            tagComp.method_57447(class_2509.field_11560, CTPData.CODEC, Optional.of(ctpd)).getOrThrow());
        return stack;
    }

    @Nullable
    public static CustomTooltipProvider<?> getProvider(class_1799 stack){
        var tagComp = stack.method_57824(class_9334.field_49628);
        if(tagComp == null) return null;
        try {
            return tagComp.method_57446(CTPData.CODEC).getOrThrow().map(CTPData::provider).orElse(null);
        } catch (Exception e){
            return null;
        }
    }

    @SuppressWarnings("unchecked")
    @Nullable
    public static List<class_2561> getTooltipText(class_1799 stack){
        var tagComp = stack.method_57824(class_9334.field_49628);
        if(tagComp == null) return null;
        try {
            return tagComp.method_57446(CTPData.CODEC).getOrThrow()
                .map(ctpd -> ctpd.provider().getTooltipText(ctpd.data))
                .orElse(null);
        } catch (Exception e){
            return null;
        }
    }

    @SuppressWarnings("unchecked")
    @Nullable
    public static <T> Optional<class_5632> getTooltipData(class_1799 stack){
        var tagComp = stack.method_57824(class_9334.field_49628);
        if(tagComp == null) return Optional.empty();
        try {
            return tagComp.method_57446(CTPData.CODEC).getOrThrow()
                .flatMap(ctpd -> ctpd.provider().getTooltipData(ctpd.data));
        } catch (Exception e){
            return Optional.empty();
        }
    }

    private static final Map<class_2960, CustomTooltipProvider> PROVIDERS = new HashMap<>();

    private static final Codec<CustomTooltipProvider> PROVIDERS_CODEC = class_2960.field_25139.comapFlatMap(
        DataResult.partialGet(PROVIDERS::get, () -> "Provider not found"),
        CustomTooltipProvider::getId
    );

    /**
     * Registers a provider.
     * @param <T> type that the provider handles
     * @param provider
     * @return the same provider.
     */
    public static <T> CustomTooltipProvider<T> registerProvider(CustomTooltipProvider<T> provider){
        PROVIDERS.put(provider.getId(), provider);
        return provider;
    }

    /**
     * Makes a tooltip out of some arbitrary data of type T.
     * <p>
     * Delegating between providers is handled by the manager.
     */
    public static interface CustomTooltipProvider<T>{

        public class_2960 getId();

        @NotNull
        public List<class_2561> getTooltipText(T content);

        @NotNull
        public Optional<class_5632> getTooltipData(T content);

        @NotNull
        public Codec<T> getCodec();
    }

    // helper class to make data storage a tad easier
    public record CTPData<T>(CustomTooltipProvider<T> provider, T data){
        @SuppressWarnings("unchecked")
        public static final MapCodec<Optional<CTPData>> CODEC = PROVIDERS_CODEC.dispatch("inline_ctp_id",
            (CTPData ctpd) -> ctpd.provider(),
            CTPData::makeCTPDCodec
        ).optionalFieldOf(TOOLTIP_DATA_KEY);

        public static <T> MapCodec<CTPData<T>> makeCTPDCodec(CustomTooltipProvider<T> provider){
            return provider.getCodec().fieldOf("inline_ctp_data").xmap(
                d -> new CTPData<T>(provider, d),
                CTPData::data
            );
        }
    }
}
