package com.samsthenerd.inline.tooltips;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import javax.annotation.Nullable;
import net.minecraft.class_1792;
import net.minecraft.class_1799;
import net.minecraft.class_1802;
import net.minecraft.class_2487;
import net.minecraft.class_2509;
import net.minecraft.class_2561;
import net.minecraft.class_2568;
import net.minecraft.class_2960;
import net.minecraft.class_5632;
import net.minecraft.class_5684;
import org.jetbrains.annotations.NotNull;

import com.mojang.serialization.Codec;
import com.mojang.serialization.DataResult;
import com.samsthenerd.inline.Inline;

/**
 * An item-less custom tooltip system.
 * <p>
 * By default Minecraft only supports 3 types of {@link class_2568}s:
 * Text, Item, and Entity (which is kinda just text again). But, Item tooltips
 * have a pretty good tooltip system with {@link class_5632} and {@link class_5684}s.
 * Unfortunately, using those requires adding a new item, which is rather inconvenient, so
 * instead we hijack an arbitrary vanilla item to throw our tooltip system on. 
 * <p>
 * You can use these tooltips in hover events by making an item based hover event
 * out of the dummy itemstack from {@link CustomTooltipManager#getForTooltip(CustomTooltipProvider, Object)}.
 * <p>
 * Note that this isn't necessarily a part of the core Inline API, just a useful system
 * used for the built in player facing features.
 * @see CustomTooltipProvider
 */
public class CustomTooltipManager {

    public static final class_1792 HIJACKED_ITEM = class_1802.field_8153;

    /**
     * Gets a dummy itemstack hijacked with the data to display the tooltip 
     * provided by the provider for the given content. 
     * @param <T> Type of the content that the provider handles
     * @param provider A tooltip provider that can handle the given content.
     * @param content Arbitrary data to be handled by the provider.
     * @return an itemstack that will have a tooltip given by our provider and content.
     */
    public static <T> class_1799 getForTooltip(CustomTooltipProvider<T> provider, T content){
        class_1799 stack = new class_1799(HIJACKED_ITEM);
        class_2487 tag = new class_2487();
        tag.method_10582("id", provider.getId().toString());
        tag.method_10566("data", provider.getCodec().encodeStart(class_2509.field_11560, content).getOrThrow(false, Inline.LOGGER::error));
        stack.method_7959("inlinecustomtooltip", tag);
        return stack;
    }

    @Nullable
    public static CustomTooltipProvider<?> getProvider(class_1799 stack){
        class_2487 tag = stack.method_7941("inlinecustomtooltip");
        // we're throwing this all in a try/catch because it's good enough 
        try{
            return PROVIDERS.get(new class_2960(tag.method_10558("id")));
        } catch(Exception e) {
            return null;
        }
    }

    @Nullable
    public static <T> List<class_2561> getTooltipText(class_1799 stack){
        try{
            class_2487 tag = stack.method_7941("inlinecustomtooltip");
            CustomTooltipProvider<T> provider = (CustomTooltipProvider<T>)getProvider(stack);
            DataResult<T> contentRes = provider.getCodec().parse(class_2509.field_11560, tag.method_10580("data"));
            T content = contentRes.resultOrPartial(Inline.LOGGER::error).orElseThrow();
            return provider.getTooltipText(content);
        } catch(Exception e) {
            return null;
        }
    }

    @Nullable
    public static <T> Optional<class_5632> getTooltipData(class_1799 stack){
        try{
            class_2487 tag = stack.method_7941("inlinecustomtooltip");
            CustomTooltipProvider<T> provider = (CustomTooltipProvider<T>)getProvider(stack);
            DataResult<T> contentRes = provider.getCodec().parse(class_2509.field_11560, tag.method_10580("data"));
            T content = contentRes.resultOrPartial(Inline.LOGGER::error).orElseThrow();
            return provider.getTooltipData(content);
        } catch(Exception e) {
            return null;
        }
    }

    private static final Map<class_2960, CustomTooltipProvider> PROVIDERS = new HashMap<>();

    /**
     * Registers a provider.
     * @param <T> type that the provider handles
     * @param provider
     * @return the same provider.
     */
    public static <T> CustomTooltipProvider<T> registerProvider(CustomTooltipProvider<T> provider){
        PROVIDERS.put(provider.getId(), provider);
        return provider;
    }

    /**
     * Makes a tooltip out of some arbitrary data of type T.
     * <p>
     * Delegating between providers is handled by the manager.
     */
    public static interface CustomTooltipProvider<T>{

        public class_2960 getId();

        @NotNull
        public List<class_2561> getTooltipText(T content);

        @NotNull
        public Optional<class_5632> getTooltipData(T content);

        @NotNull
        public Codec<T> getCodec();
    }
}
