package com.almostreliable.unified.unification;

import com.almostreliable.unified.AlmostUnifiedCommon;
import com.almostreliable.unified.api.unification.TagSubstitutions;
import com.almostreliable.unified.utils.VanillaTagWrapper;

import com.google.common.collect.HashMultimap;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableMultimap;
import com.google.common.collect.Multimap;

import org.jetbrains.annotations.Nullable;

import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import net.minecraft.class_1792;
import net.minecraft.class_2960;
import net.minecraft.class_6862;
import net.minecraft.class_6880;
import net.minecraft.class_7924;

public final class TagSubstitutionsImpl implements TagSubstitutions {

    private final Map<class_6862<class_1792>, class_6862<class_1792>> replacedToSubstitute;
    private final Multimap<class_6862<class_1792>, class_6862<class_1792>> substituteToReplaced;

    private TagSubstitutionsImpl(Map<class_6862<class_1792>, class_6862<class_1792>> replacedToSubstitute, Multimap<class_6862<class_1792>, class_6862<class_1792>> substituteToReplaced) {
        this.replacedToSubstitute = replacedToSubstitute;
        this.substituteToReplaced = substituteToReplaced;
    }

    /**
     * Creates a new tag substitutions instance that contains immutable maps of all tag substitution relationships.
     * <p>
     * This method ensures that all substitute tags are unify tags and that all replaced tags are not unify tags.<br>
     * Since substitute tags have to be unify-tags, it is ensured that they are valid and exist in the game. Replaced
     * tags still need to be validated.
     *
     * @param validTagFilter        a filter that defines which tags are valid tags and exist in the game
     * @param unifyTagFilter        a filter that defines which tags are unify tags
     * @param configuredSubstitutes the tag substitution relationships from the config
     * @return the new tag substitutions instance
     */
    public static TagSubstitutionsImpl create(Predicate<class_6862<class_1792>> validTagFilter, Predicate<class_6862<class_1792>> unifyTagFilter, Map<class_2960, Set<class_2960>> configuredSubstitutes) {
        ImmutableMap.Builder<class_6862<class_1792>, class_6862<class_1792>> refsToSubsBuilder = ImmutableMap.builder();
        ImmutableMultimap.Builder<class_6862<class_1792>, class_6862<class_1792>> subsToRefsBuilder = ImmutableMultimap.builder();
        Set<class_6862<class_1792>> invalidReplacedTags = new HashSet<>();
        Set<class_6862<class_1792>> unifyReplacedTags = new HashSet<>();

        configuredSubstitutes.forEach((rawSubstituteTag, rawReplacedTags) -> {
            for (class_2960 rawReplacedTag : rawReplacedTags) {
                var substituteTag = class_6862.method_40092(class_7924.field_41197, rawSubstituteTag);
                var replacedTag = class_6862.method_40092(class_7924.field_41197, rawReplacedTag);

                if (!unifyTagFilter.test(substituteTag)) {
                    AlmostUnifiedCommon.LOGGER.warn(
                        "[TagSubstitutions] Substitute tag '#{}' is not configured as a unify tag! Config entry '#{} -> {}' will be ignored.",
                        substituteTag.comp_327(),
                        substituteTag.comp_327(),
                        rawReplacedTags.stream().map(t -> "#" + t).collect(Collectors.joining(", "))
                    );
                    return; // don't check other replaced tags if the substitute tag is invalid
                }

                if (!validTagFilter.test(replacedTag)) {
                    invalidReplacedTags.add(replacedTag);
                    continue; // only skip the current invalid replaced tag
                }

                if (unifyTagFilter.test(replacedTag)) {
                    unifyReplacedTags.add(replacedTag);
                    continue; // only skip the current invalid replaced tag
                }

                refsToSubsBuilder.put(replacedTag, substituteTag);
                subsToRefsBuilder.put(substituteTag, replacedTag);
            }

            if (!invalidReplacedTags.isEmpty()) {
                AlmostUnifiedCommon.LOGGER.warn(
                    "[TagSubstitutions] Substitute tag '#{}' contains invalid replaced tags! Affected tags: {}",
                    rawSubstituteTag,
                    invalidReplacedTags.stream().map(t -> "#" + t.comp_327()).collect(Collectors.joining(", "))
                );
            }

            if (!unifyReplacedTags.isEmpty()) {
                AlmostUnifiedCommon.LOGGER.warn(
                    "[TagSubstitutions] Substitute tag '#{}' contains replaced tags that are configured as unify tags! Affected tags: {}",
                    rawSubstituteTag,
                    unifyReplacedTags.stream().map(t -> "#" + t.comp_327()).collect(Collectors.joining(", "))
                );
            }
        });

        return new TagSubstitutionsImpl(refsToSubsBuilder.build(), subsToRefsBuilder.build());
    }

    /**
     * Applies tag substitutions to the provided item tags.
     *
     * @param itemTags the item tags to apply the substitutions to
     */
    public void apply(VanillaTagWrapper<class_1792> itemTags) {
        Multimap<class_2960, class_2960> changedTags = HashMultimap.create();

        substituteToReplaced.asMap().forEach((substituteTag, replacedTags) -> {
            for (var replacedTag : replacedTags) {
                var replacedTagHolders = itemTags.get(replacedTag.comp_327());
                for (var replacedTagHolder : replacedTagHolders) {
                    itemTags.add(substituteTag.comp_327(), replacedTagHolder);
                    replacedTagHolder
                        .method_40230()
                        .ifPresent(key -> changedTags.put(substituteTag.comp_327(), key.method_29177()));
                }
            }
        });

        changedTags.asMap().forEach((tag, entries) -> AlmostUnifiedCommon.LOGGER.info(
            "[TagSubstitutions] Added items of replaced tags to substitute tag '#{}'. Added items: {}",
            tag,
            entries
        ));
    }


    @Override
    @Nullable
    public class_6862<class_1792> getSubstituteTag(class_6862<class_1792> replacedTag) {
        return replacedToSubstitute.get(replacedTag);
    }

    @Override
    public Collection<class_6862<class_1792>> getReplacedTags(class_6862<class_1792> substituteTag) {
        return Collections.unmodifiableCollection(substituteToReplaced.get(substituteTag));
    }

    @Override
    public Set<class_6862<class_1792>> getReplacedTags() {
        return replacedToSubstitute.keySet();
    }
}
