package com.almostreliable.unified.recipe;

import com.almostreliable.unified.BuildConfig;
import com.almostreliable.unified.utils.Utils;
import com.google.gson.JsonArray;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import javax.annotation.Nullable;
import net.minecraft.class_1263;
import net.minecraft.class_1799;
import net.minecraft.class_1860;
import net.minecraft.class_1865;
import net.minecraft.class_1937;
import net.minecraft.class_2540;
import net.minecraft.class_2960;
import net.minecraft.class_3956;
import net.minecraft.class_5455;
import java.util.HashMap;
import java.util.Map;

/**
 * This recipe is used to track which recipes were unified. It is NOT used for crafting.
 * Each tracker will hold one namespace with a list of recipes that were unified for it.
 */
public class ClientRecipeTracker implements class_1860<class_1263> {
    public static final class_2960 ID = Utils.getRL("client_recipe_tracker");
    public static final String RECIPES = "recipes";
    public static final String NAMESPACE = "namespace";
    public static final int UNIFIED_FLAG = 1;
    public static final int DUPLICATE_FLAG = 2;
    public static final class_1865<ClientRecipeTracker> SERIALIZER = new Serializer();
    public static final class_3956<ClientRecipeTracker> TYPE = new class_3956<>() {
        @Override
        public String toString() {
            return ID.method_12832();
        }
    };

    private final class_2960 id;
    private final Map<class_2960, ClientRecipeLink> recipes = new HashMap<>();
    private final String namespace;

    protected ClientRecipeTracker(class_2960 id, String namespace) {
        this.id = id;
        this.namespace = namespace;
    }

    /**
     * Creates a raw string representation.
     *
     * @param isUnified   Whether the recipe was unified.
     * @param isDuplicate Whether the recipe had duplicates.
     * @param idPath      The path of the recipe.
     * @return String representation as: `flag$idPath`
     */
    private static String createRaw(boolean isUnified, boolean isDuplicate, String idPath) {
        int flag = 0;
        if (isUnified) flag |= UNIFIED_FLAG;
        if (isDuplicate) flag |= DUPLICATE_FLAG;
        return flag + "$" + idPath;
    }

    //<editor-fold defaultstate="collapsed" desc="Default recipe stuff. Ignore this. Forget this.">
    @Override
    public boolean method_8115(class_1263 container, class_1937 level) {
        return false;
    }

    @Override
    public class_1799 method_8116(class_1263 container, class_5455 registryAccess) {
        return class_1799.field_8037;
    }

    @Override
    public boolean method_8113(int width, int height) {
        return false;
    }

    @Override
    public class_1799 method_8110(class_5455 registryAccess) {
        return class_1799.field_8037;
    }

    @Override
    public class_2960 method_8114() {
        return id;
    }
    //</editor-fold>

    @Override
    public class_1865<?> method_8119() {
        return SERIALIZER;
    }

    @Override
    public class_3956<?> method_17716() {
        return TYPE;
    }

    private void add(ClientRecipeLink clientRecipeLink) {
        recipes.put(clientRecipeLink.id(), clientRecipeLink);
    }

    @Nullable
    public ClientRecipeLink getLink(class_2960 recipeId) {
        return recipes.get(recipeId);
    }

    public record ClientRecipeLink(class_2960 id, boolean isUnified, boolean isDuplicate) {}

    public static class Serializer implements class_1865<ClientRecipeTracker> {

        /**
         * Reads a recipe from a json file. Recipe will look like this:
         * <pre>
         * {@code
         * {
         *      "type": "almostunified:client_recipe_tracker",
         *      "namespace": "minecraft", // The namespace of the recipes.
         *      "recipes": [
         *          "flag$recipePath",
         *          "flag$recipe2Path",
         *          ...
         *          "flag$recipeNPath"
         *      ]
         * }
         * }
         * </pre>
         *
         * @param recipeId The id of the recipe for the tracker.
         * @param json     The json object.
         * @return The recipe tracker.
         */
        @Override
        public ClientRecipeTracker method_8121(class_2960 recipeId, JsonObject json) {
            String namespace = json.get(NAMESPACE).getAsString();
            JsonArray recipes = json.get(RECIPES).getAsJsonArray();
            ClientRecipeTracker tracker = new ClientRecipeTracker(recipeId, namespace);
            for (JsonElement element : recipes) {
                ClientRecipeLink clientRecipeLink = parseRaw(namespace, element.getAsString());
                tracker.add(clientRecipeLink);
            }
            return tracker;
        }

        @Override
        public ClientRecipeTracker method_8122(class_2960 recipeId, class_2540 buffer) {
            int size = buffer.readInt();
            String namespace = buffer.method_19772();

            ClientRecipeTracker recipe = new ClientRecipeTracker(recipeId, namespace);
            for (int i = 0; i < size; i++) {
                String raw = buffer.method_19772();
                ClientRecipeLink clientRecipeLink = parseRaw(namespace, raw);
                recipe.add(clientRecipeLink);
            }
            return recipe;
        }

        /**
         * Writes the tracker to the buffer. The namespace is written separately to save some bytes.
         * Buffer output will look like:
         * <pre>
         *     size
         *     namespace
         *     flag$recipePath
         *     flag$recipe2Path
         *     ...
         *     flag$recipeNPath
         * </pre>
         *
         * @param buffer The buffer to write to
         * @param recipe The recipe to write
         */
        @Override
        public void toNetwork(class_2540 buffer, ClientRecipeTracker recipe) {
            buffer.writeInt(recipe.recipes.size());
            buffer.method_10814(recipe.namespace);
            for (ClientRecipeLink clientRecipeLink : recipe.recipes.values()) {
                String raw = createRaw(clientRecipeLink.isUnified(),
                        clientRecipeLink.isDuplicate(),
                        clientRecipeLink.id().method_12832());
                buffer.method_10814(raw);
            }
        }

        /**
         * Creates a {@link ClientRecipeLink} from a raw string for the given namespace.
         *
         * @param namespace The namespace to use.
         * @param raw       The raw string.
         * @return The client sided recipe link.
         */
        private static ClientRecipeLink parseRaw(String namespace, String raw) {
            String[] split = raw.split("\\$", 2);
            int flag = Integer.parseInt(split[0]);
            boolean isUnified = (flag & UNIFIED_FLAG) != 0;
            boolean isDuplicate = (flag & DUPLICATE_FLAG) != 0;
            return new ClientRecipeLink(new class_2960(namespace, split[1]), isUnified, isDuplicate);
        }
    }

    public static class RawBuilder {

        private final Map<String, JsonArray> recipesByNamespace = new HashMap<>();

        public void add(RecipeLink recipe) {
            class_2960 recipeId = recipe.getId();
            JsonArray array = recipesByNamespace.computeIfAbsent(recipeId.method_12836(), k -> new JsonArray());
            array.add(createRaw(recipe.isUnified(), recipe.hasDuplicateLink(), recipeId.method_12832()));
        }

        /**
         * Creates a map with the namespace as key and the json recipe.
         * These recipes are used later in {@link Serializer#method_8121(class_2960, JsonObject)}
         *
         * @return The map with the namespace as key and the json recipe.
         */
        public Map<class_2960, JsonObject> compute() {
            Map<class_2960, JsonObject> result = new HashMap<>();
            recipesByNamespace.forEach((namespace, recipes) -> {
                JsonObject json = new JsonObject();
                json.addProperty("type", ID.toString());
                json.addProperty(NAMESPACE, namespace);
                json.add(RECIPES, recipes);
                result.put(new class_2960(BuildConfig.MOD_ID, namespace), json);
            });
            return result;
        }
    }
}
