package com.almostreliable.unified.unification;

import com.almostreliable.unified.api.unification.ModPriorities;
import com.almostreliable.unified.api.unification.StoneVariants;
import com.almostreliable.unified.api.unification.TagSubstitutions;
import com.almostreliable.unified.api.unification.UnificationEntry;
import com.almostreliable.unified.api.unification.UnificationLookup;
import com.almostreliable.unified.api.unification.UnificationSettings;
import com.almostreliable.unified.config.UnificationConfig;
import com.almostreliable.unified.unification.UnificationLookupImpl.Builder;
import com.almostreliable.unified.utils.VanillaTagWrapper;

import org.jetbrains.annotations.Nullable;

import java.util.Collection;
import java.util.List;
import java.util.function.Function;
import java.util.function.Predicate;
import net.minecraft.class_1792;
import net.minecraft.class_1799;
import net.minecraft.class_1856;
import net.minecraft.class_2248;
import net.minecraft.class_2960;
import net.minecraft.class_6862;
import net.minecraft.class_6880;

public final class UnificationSettingsImpl implements UnificationSettings {

    private final String name;
    private final ModPriorities modPriorities;
    private final StoneVariants stoneVariants;
    private final Function<class_2960, Boolean> recipeTypeCheck;
    private final Function<class_2960, Boolean> recipeIdCheck;
    private final boolean recipeViewerHiding;
    private final boolean lootUnification;
    private final Function<class_2960, Boolean> lootTableCheck;
    private final UnificationLookup unificationLookup;
    private final Runnable clearCaches;

    private UnificationSettingsImpl(String name, ModPriorities modPriorities, StoneVariants stoneVariants, Function<class_2960, Boolean> recipeTypeCheck, Function<class_2960, Boolean> recipeIdCheck, boolean recipeViewerHiding, boolean lootUnification, Function<class_2960, Boolean> lootTableCheck, UnificationLookup unificationLookup, Runnable clearCaches) {
        this.name = name;
        this.modPriorities = modPriorities;
        this.stoneVariants = stoneVariants;
        this.recipeTypeCheck = recipeTypeCheck;
        this.recipeIdCheck = recipeIdCheck;
        this.recipeViewerHiding = recipeViewerHiding;
        this.lootUnification = lootUnification;
        this.lootTableCheck = lootTableCheck;
        this.unificationLookup = unificationLookup;
        this.clearCaches = clearCaches;
    }

    public static List<UnificationSettings> create(Collection<UnificationConfig> configs, VanillaTagWrapper<class_1792> itemTags, VanillaTagWrapper<class_2248> blockTags, TagSubstitutionsImpl tagSubstitutions) {
        return configs
            .stream()
            .map(config -> create(config, itemTags, blockTags, tagSubstitutions))
            .toList();
    }

    public static UnificationSettings create(UnificationConfig config, VanillaTagWrapper<class_1792> itemTags, VanillaTagWrapper<class_2248> blockTags, TagSubstitutions tagSubstitutions) {
        var lookupBuilder = new UnificationLookupImpl.Builder();
        for (var tag : config.getTags()) {
            var itemHolders = itemTags.get(tag);
            for (var itemHolder : itemHolders) {
                itemHolder.method_40230().ifPresent(itemKey -> {
                    var itemId = itemKey.method_29177();
                    if (config.shouldIgnoreItem(itemId)) return;
                    lookupBuilder.put(tag, itemId);
                });
            }
        }

        ModPriorities modPriorities = config.getModPriorities();
        StoneVariants stoneVariants = StoneVariantsImpl.create(
            config.getStoneVariants(),
            itemTags,
            blockTags
        );

        return new UnificationSettingsImpl(
            config.getName(),
            modPriorities,
            stoneVariants,
            config::shouldIncludeRecipeType,
            config::shouldIncludeRecipeId,
            config.shouldHideVariantItems(),
            config.shouldUnifyLoot(),
            config::shouldIncludeLootTable,
            lookupBuilder.build(modPriorities, stoneVariants, tagSubstitutions),
            config::clearCaches
        );
    }

    @Override
    public String getName() {
        return name;
    }

    @Override
    public ModPriorities getModPriorities() {
        return modPriorities;
    }

    @Override
    public StoneVariants getStoneVariants() {
        return stoneVariants;
    }

    @Override
    public boolean shouldIncludeRecipeType(class_2960 type) {
        return recipeTypeCheck.apply(type);
    }

    @Override
    public boolean shouldIncludeRecipeId(class_2960 id) {
        return recipeIdCheck.apply(id);
    }

    @Override
    public boolean shouldHideVariantItems() {
        return recipeViewerHiding;
    }

    @Override
    public boolean shouldUnifyLoot() {
        return lootUnification;
    }

    @Override
    public boolean shouldIncludeLootTable(class_2960 table) {
        return lootTableCheck.apply(table);
    }

    @Override
    public Collection<class_6862<class_1792>> getTags() {
        return unificationLookup.getTags();
    }

    @Override
    public Collection<UnificationEntry<class_1792>> getTagEntries(class_6862<class_1792> tag) {
        return unificationLookup.getTagEntries(tag);
    }

    @Nullable
    @Override
    public UnificationEntry<class_1792> getItemEntry(class_2960 item) {
        return unificationLookup.getItemEntry(item);
    }

    @Nullable
    @Override
    public class_6862<class_1792> getRelevantItemTag(class_2960 item) {
        return unificationLookup.getRelevantItemTag(item);
    }

    @Nullable
    @Override
    public UnificationEntry<class_1792> getVariantItemTarget(class_2960 item) {
        return unificationLookup.getVariantItemTarget(item);
    }

    @Nullable
    @Override
    public UnificationEntry<class_1792> getTagTargetItem(class_6862<class_1792> tag, Predicate<class_2960> itemFilter) {
        return unificationLookup.getTagTargetItem(tag, itemFilter);
    }

    @Override
    public boolean isUnifiedIngredientItem(class_1856 ingredient, class_1799 item) {
        return unificationLookup.isUnifiedIngredientItem(ingredient, item);
    }

    public void clearCache() {
        clearCaches.run();
    }
}
