package com.almostreliable.unified.api;

import com.almostreliable.unified.api.unification.Placeholders;

import org.jetbrains.annotations.Nullable;

import java.util.Collection;
import java.util.ServiceLoader;
import java.util.Set;
import net.minecraft.class_1792;
import net.minecraft.class_1935;
import net.minecraft.class_6862;

/**
 * The core interface for the Almost Unified API.
 * <p>
 * Use this to get an instance of the {@link AlmostUnifiedRuntime} or to look up unification information.
 *
 * @since 1.0.0
 */
public interface AlmostUnified {

    /**
     * The default instance of Almost Unified.
     * <p>
     * If unavailable, it will return an empty instance that only returns default values for each method.<br>
     * This instance is only available on the logical server side.
     */
    @SuppressWarnings("InnerClassReferencedViaSubclass")
    AlmostUnified INSTANCE = ServiceLoader.load(AlmostUnified.class).findFirst().orElseGet(Empty::new);

    /**
     * Returns whether the {@link AlmostUnifiedRuntime} is loaded and ready to be used.
     * <p>
     * The {@link AlmostUnifiedRuntime} is only available on the logical server side.
     *
     * @return true if the {@link AlmostUnifiedRuntime} is loaded, false otherwise
     */
    boolean isRuntimeLoaded();

    /**
     * Returns the instance of the {@link AlmostUnifiedRuntime}.
     * <p>
     * The {@link AlmostUnifiedRuntime} is only available on the logical server side. This method returns null if the
     * runtime is not loaded. To check this beforehand, use {@link #isRuntimeLoaded()}. If you are sure the runtime is
     * available, you can use {@link #getRuntimeOrThrow()} instead.
     *
     * @return the {@link AlmostUnifiedRuntime}, or null if the runtime is not loaded
     */
    @Nullable
    AlmostUnifiedRuntime getRuntime();

    /**
     * Returns the instance of the {@link AlmostUnifiedRuntime}.
     * <p>
     * The {@link AlmostUnifiedRuntime} is only available on the logical server side. This method throws an exception
     * if the runtime is not loaded. To check this beforehand, use {@link #isRuntimeLoaded()}.
     *
     * @return the {@link AlmostUnifiedRuntime}
     */
    AlmostUnifiedRuntime getRuntimeOrThrow();

    /**
     * Returns all {@link class_6862}s used for the unification process.
     * <p>
     * The returned collection will only contain tags that have their {@link Placeholders} replaced and have been
     * validated. All tags will be unique.
     *
     * @return the {@link class_6862}s used for the unification, or empty if no tags are used
     */
    Collection<class_6862<class_1792>> getTags();

    /**
     * Returns all item entries for the given {@link class_6862}.
     * <p>
     * The returned collection will only contain entries if the provided {@link class_6862} is a configured unification tag.
     *
     * @param tag the {@link class_6862} to get the entries for
     * @return the item entries for the {@link class_6862}, or empty if not found
     */
    Collection<class_1792> getTagEntries(class_6862<class_1792> tag);

    /**
     * Returns the relevant {@link class_6862} for the given {@link class_1935}
     * <p>
     * Since an item can only have a single relevant tag, this method is guaranteed to return a single {@link class_6862} as
     * long as there exists a configured unification tag that includes the item.
     *
     * @param itemLike the {@link class_1935} to get the relevant {@link class_6862} for
     * @return the relevant {@link class_6862}, or null if not found
     */
    @Nullable
    class_6862<class_1792> getRelevantItemTag(class_1935 itemLike);

    /**
     * Returns the target item for the given variant {@link class_1935}.
     * <p>
     * The target item describes the item with the highest priority among all variant items within a {@link class_6862}.
     * It is used to replace the variant items in the unification process.<br>
     * This method will return null if no configured unification tag exists that includes the given item.
     * <p>
     * If the item is part of a stone variant, it will only check items within the same stone variant.
     *
     * @param itemLike the variant {@link class_1935} to get the target item for
     * @return the target item, or null if not found
     */
    @Nullable
    class_1792 getVariantItemTarget(class_1935 itemLike);

    /**
     * Returns the target item for the given {@link class_6862}.
     * <p>
     * The target item describes the item with the highest priority among all variant items within a {@link class_6862}.
     * It is used to replace the variant items in the unification process.<br>
     * This method will return null the given {@link class_6862} is not a configured unification tag.
     *
     * @param tag the {@link class_6862} to get the target item for
     * @return the target item, or null if not found
     */
    @Nullable
    class_1792 getTagTargetItem(class_6862<class_1792> tag);

    class Empty implements AlmostUnified {

        @Override
        public boolean isRuntimeLoaded() {
            return false;
        }

        @Nullable
        @Override
        public AlmostUnifiedRuntime getRuntime() {
            return null;
        }

        @Override
        public AlmostUnifiedRuntime getRuntimeOrThrow() {
            throw new IllegalStateException("runtime is not loaded");
        }

        @Override
        public Collection<class_6862<class_1792>> getTags() {
            return Set.of();
        }

        @Override
        public Collection<class_1792> getTagEntries(class_6862<class_1792> tag) {
            return Set.of();
        }

        @Nullable
        @Override
        public class_6862<class_1792> getRelevantItemTag(class_1935 itemLike) {
            return null;
        }

        @Nullable
        @Override
        public class_1792 getVariantItemTarget(class_1935 itemLike) {
            return null;
        }

        @Nullable
        @Override
        public class_1792 getTagTargetItem(class_6862<class_1792> tag) {
            return null;
        }
    }
}
