package com.almostreliable.unified.unification;

import com.almostreliable.unified.AlmostUnifiedCommon;
import com.almostreliable.unified.api.unification.ModPriorities;
import com.almostreliable.unified.api.unification.StoneVariants;
import com.almostreliable.unified.api.unification.TagSubstitutions;
import com.almostreliable.unified.api.unification.UnificationEntry;
import com.almostreliable.unified.api.unification.UnificationLookup;

import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;

import org.jetbrains.annotations.Nullable;

import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Predicate;
import net.minecraft.class_1792;
import net.minecraft.class_1799;
import net.minecraft.class_1856;
import net.minecraft.class_2960;
import net.minecraft.class_6862;
import net.minecraft.class_7923;

public final class UnificationLookupImpl implements UnificationLookup {

    private final ModPriorities modPriorities;
    private final StoneVariants stoneVariants;
    private final TagSubstitutions tagSubstitutions;
    private final Map<class_6862<class_1792>, Set<UnificationEntry<class_1792>>> tagsToEntries;
    private final Map<class_2960, UnificationEntry<class_1792>> idsToEntries;

    private UnificationLookupImpl(ModPriorities modPriorities, StoneVariants stoneVariants, TagSubstitutions tagSubstitutions, Map<class_6862<class_1792>, Set<UnificationEntry<class_1792>>> tagsToEntries, Map<class_2960, UnificationEntry<class_1792>> idsToEntries) {
        this.modPriorities = modPriorities;
        this.stoneVariants = stoneVariants;
        this.tagSubstitutions = tagSubstitutions;
        this.tagsToEntries = tagsToEntries;
        this.idsToEntries = idsToEntries;
    }

    @Override
    public Collection<class_6862<class_1792>> getTags() {
        return tagsToEntries.keySet();
    }

    @Override
    public Collection<UnificationEntry<class_1792>> getTagEntries(class_6862<class_1792> tag) {
        return tagsToEntries.getOrDefault(tag, Collections.emptySet());
    }

    @Nullable
    @Override
    public UnificationEntry<class_1792> getItemEntry(class_2960 item) {
        return idsToEntries.get(item);
    }

    @Nullable
    @Override
    public class_6862<class_1792> getRelevantItemTag(class_2960 item) {
        UnificationEntry<class_1792> entry = idsToEntries.get(item);
        return entry == null ? null : entry.tag();
    }

    @Nullable
    @Override
    public UnificationEntry<class_1792> getVariantItemTarget(class_2960 item) {
        var tag = getRelevantItemTag(item);
        if (tag == null) return null;

        if (stoneVariants.isOreTag(tag)) {
            String stoneVariant = stoneVariants.getStoneVariant(item);
            return getTagTargetItem(tag, itemId -> stoneVariant.equals(stoneVariants.getStoneVariant(itemId)));
        }

        return getTagTargetItem(tag);
    }

    @Nullable
    @Override
    public UnificationEntry<class_1792> getTagTargetItem(class_6862<class_1792> tag, Predicate<class_2960> itemFilter) {
        var substituteTag = tagSubstitutions.getSubstituteTag(tag);
        var tagToCheck = substituteTag != null ? substituteTag : tag;

        var items = getTagEntries(tagToCheck)
            .stream()
            .filter(entry -> itemFilter.test(entry.id()))
            // sort by length so clean stone variants come first
            .sorted(Comparator.comparingInt(value -> value.id().toString().length()))
            .toList();

        return items.isEmpty() ? null : modPriorities.findTargetItem(tagToCheck, items);
    }

    @Override
    public boolean isUnifiedIngredientItem(class_1856 ingredient, class_1799 item) {
        Set<class_6862<class_1792>> checkedTags = new HashSet<>();

        for (class_1799 stack : ingredient.method_8105()) {
            class_2960 itemId = class_7923.field_41178.method_10221(stack.method_7909());

            var relevantTag = getRelevantItemTag(itemId);
            if (relevantTag == null || checkedTags.contains(relevantTag)) continue;
            checkedTags.add(relevantTag);

            if (item.method_31573(relevantTag)) {
                return true;
            }
        }

        return false;
    }

    public static class Builder {

        private final Map<UnificationEntry<class_1792>, class_6862<class_1792>> entriesToTags = new HashMap<>();
        private final Map<class_6862<class_1792>, Set<UnificationEntry<class_1792>>> tagsToEntries = new HashMap<>();

        private void put(class_6862<class_1792> tag, UnificationEntry<class_1792> entry) {
            if (entriesToTags.containsKey(entry)) {
                var boundTag = entriesToTags.get(entry);
                AlmostUnifiedCommon.LOGGER.error(
                    "Unification entry for item '{}' with tag '#{}' is already part of tag '#{}'.",
                    entry.id(),
                    tag.comp_327(),
                    boundTag.comp_327()
                );
                return;
            }

            entriesToTags.put(entry, tag);
            tagsToEntries.computeIfAbsent(tag, $ -> new HashSet<>()).add(entry);
        }

        public Builder put(class_6862<class_1792> tag, class_2960... ids) {
            for (class_2960 id : ids) {
                put(tag, new UnificationEntryImpl<>(class_7923.field_41178, id));
            }

            return this;
        }

        public Builder put(class_6862<class_1792> tag, class_1792... items) {
            for (var item : items) {
                put(tag, new UnificationEntryImpl<>(class_7923.field_41178, item));
            }

            return this;
        }

        public UnificationLookup build(ModPriorities modPriorities, StoneVariants stoneVariants, TagSubstitutions tagSubstitutions) {
            ImmutableMap.Builder<class_6862<class_1792>, Set<UnificationEntry<class_1792>>> tagsToEntriesBuilder = ImmutableMap.builder();
            ImmutableMap.Builder<class_2960, UnificationEntry<class_1792>> idsToEntriesBuilder = ImmutableMap.builder();

            tagsToEntries.forEach((tag, entries) -> {
                ImmutableSet.Builder<UnificationEntry<class_1792>> entrySetBuilder = ImmutableSet.builder();
                for (var entry : entries) {
                    entrySetBuilder.add(entry);
                    ((UnificationEntryImpl<class_1792>) entry).bindTag(tag);
                    idsToEntriesBuilder.put(entry.id(), entry);
                }

                tagsToEntriesBuilder.put(tag, entrySetBuilder.build());
            });

            return new UnificationLookupImpl(
                modPriorities,
                stoneVariants,
                tagSubstitutions,
                tagsToEntriesBuilder.build(),
                idsToEntriesBuilder.build()
            );
        }
    }
}
