package com.almostreliable.unified.utils;

import com.almostreliable.unified.AlmostUnified;
import com.almostreliable.unified.ReplacementData;
import com.almostreliable.unified.config.UnifyConfig;
import com.google.common.base.Preconditions;
import com.google.common.collect.HashMultimap;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Multimap;
import javax.annotation.Nullable;
import net.minecraft.class_1792;
import net.minecraft.class_2248;
import net.minecraft.class_2960;
import net.minecraft.class_5321;
import net.minecraft.class_6880;
import net.minecraft.class_7923;
import net.minecraft.class_7924;
import java.util.Collection;
import java.util.HashSet;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Optional;
import java.util.Set;

public final class TagReloadHandler {

    private static final Object LOCK = new Object();

    private static Map<class_2960, Collection<class_6880<class_1792>>> RAW_ITEM_TAGS;
    private static Map<class_2960, Collection<class_6880<class_2248>>> RAW_BLOCK_TAGS;

    private TagReloadHandler() {}

    public static void initItemTags(Map<class_2960, Collection<class_6880<class_1792>>> rawItemTags) {
        synchronized (LOCK) {
            RAW_ITEM_TAGS = rawItemTags;
        }
    }

    public static void initBlockTags(Map<class_2960, Collection<class_6880<class_2248>>> rawBlockTags) {
        synchronized (LOCK) {
            RAW_BLOCK_TAGS = rawBlockTags;
        }
    }

    public static void run() {
        if (RAW_ITEM_TAGS == null || RAW_BLOCK_TAGS == null) {
            return;
        }

        AlmostUnified.onTagLoaderReload(RAW_ITEM_TAGS);

        RAW_ITEM_TAGS = null;
        RAW_BLOCK_TAGS = null;
    }

    public static void applyCustomTags(UnifyConfig unifyConfig) {
        Preconditions.checkNotNull(RAW_ITEM_TAGS, "Item tags were not loaded correctly");

        Multimap<class_2960, class_2960> changedItemTags = HashMultimap.create();

        for (var entry : unifyConfig.getCustomTags().entrySet()) {
            class_2960 tag = entry.getKey();
            Set<class_2960> itemIds = entry.getValue();

            for (class_2960 itemId : itemIds) {
                if (!class_7923.field_41178.method_10250(itemId)) {
                    AlmostUnified.LOG.warn("[CustomTags] Custom tag '{}' contains invalid item '{}'", tag, itemId);
                    continue;
                }

                class_5321<class_1792> itemKey = class_5321.method_29179(class_7924.field_41197, itemId);
                class_6880<class_1792> itemHolder = class_7923.field_41178.method_40264(itemKey).orElse(null);
                if (itemHolder == null) continue;

                ImmutableSet.Builder<class_6880<class_1792>> newHolders = ImmutableSet.builder();
                var currentHolders = RAW_ITEM_TAGS.get(tag);

                if (currentHolders != null) {
                    if (currentHolders.contains(itemHolder)) {
                        AlmostUnified.LOG.warn("[CustomTags] Custom tag '{}' already contains item '{}'", tag, itemId);
                        continue;
                    }

                    newHolders.addAll(currentHolders);
                }
                newHolders.add(itemHolder);

                RAW_ITEM_TAGS.put(tag, newHolders.build());
                changedItemTags.put(tag, itemId);
            }
        }

        if (!changedItemTags.isEmpty()) {
            changedItemTags.asMap().forEach((tag, items) -> {
                AlmostUnified.LOG.info("[CustomTags] Modified tag '#{}', added {}", tag, items);
            });
        }
    }

    public static boolean applyInheritance(UnifyConfig unifyConfig, ReplacementData replacementData) {
        Preconditions.checkNotNull(RAW_ITEM_TAGS, "Item tags were not loaded correctly");
        Preconditions.checkNotNull(RAW_BLOCK_TAGS, "Block tags were not loaded correctly");

        Multimap<class_2960, class_2960> changedItemTags = HashMultimap.create();
        Multimap<class_2960, class_2960> changedBlockTags = HashMultimap.create();

        var relations = resolveRelations(replacementData.filteredTagMap(), replacementData.replacementMap());
        if (relations.isEmpty()) return false;

        var blockTagMap = TagMap.createFromBlockTags(RAW_BLOCK_TAGS);
        var globalTagMap = replacementData.globalTagMap();

        for (TagRelation relation : relations) {
            var dominant = relation.dominant;
            var dominantItemHolder = findDominantItemHolder(relation);
            var dominantBlockHolder = findDominantBlockHolder(blockTagMap, dominant);

            var dominantItemTags = globalTagMap.getTagsByEntry(dominant);

            for (var item : relation.items) {
                if (dominantItemHolder != null) {
                    var changed = applyItemTags(unifyConfig, globalTagMap, dominantItemHolder, dominantItemTags, item);
                    changedItemTags.putAll(dominant, changed);
                }

                if (dominantBlockHolder != null) {
                    var changed = applyBlockTags(unifyConfig, blockTagMap, dominantBlockHolder, dominantItemTags, item);
                    changedBlockTags.putAll(dominant, changed);
                }
            }
        }

        if (!changedBlockTags.isEmpty()) {
            changedBlockTags.asMap().forEach((dominant, tags) -> {
                AlmostUnified.LOG.info("[TagInheritance] Added '{}' to block tags {}", dominant, tags);
            });
        }

        if (!changedItemTags.isEmpty()) {
            changedItemTags.asMap().forEach((dominant, tags) -> {
                AlmostUnified.LOG.info("[TagInheritance] Added '{}' to item tags {}", dominant, tags);
            });
            return true;
        }

        return false;
    }

    private static Set<TagRelation> resolveRelations(TagMap<class_1792> filteredTagMap, ReplacementMap repMap) {
        Set<TagRelation> relations = new HashSet<>();

        for (var unifyTag : filteredTagMap.getTags()) {
            var itemsByTag = filteredTagMap.getEntriesByTag(unifyTag);

            // avoid handling single entries and tags that only contain the same namespace for all items
            if (Utils.allSameNamespace(itemsByTag)) continue;

            class_2960 dominant = repMap.getPreferredItemForTag(unifyTag, $ -> true);
            if (dominant == null || !class_7923.field_41178.method_10250(dominant)) continue;

            Set<class_2960> items = getValidatedItems(itemsByTag, dominant);

            if (items.isEmpty()) continue;
            relations.add(new TagRelation(unifyTag.location(), dominant, items));
        }

        return relations;
    }

    /**
     * Returns a set of all items that are not the dominant item and are valid by checking if they are registered.
     *
     * @param itemIds  The set of all items that are in the tag
     * @param dominant The dominant item
     * @return A set of all items that are not the dominant item and are valid
     */
    private static Set<class_2960> getValidatedItems(Set<class_2960> itemIds, class_2960 dominant) {
        Set<class_2960> result = new HashSet<>(itemIds.size());
        for (class_2960 id : itemIds) {
            if (!id.equals(dominant) && class_7923.field_41178.method_10250(id)) {
                result.add(id);
            }
        }

        return result;
    }

    @SuppressWarnings("StaticVariableUsedBeforeInitialization")
    @Nullable
    private static class_6880<class_1792> findDominantItemHolder(TagRelation relation) {
        var tagHolders = RAW_ITEM_TAGS.get(relation.tag);
        if (tagHolders == null) return null;

        return findDominantHolder(tagHolders, relation.dominant);
    }

    @SuppressWarnings("StaticVariableUsedBeforeInitialization")
    @Nullable
    private static class_6880<class_2248> findDominantBlockHolder(TagMap<class_2248> tagMap, class_2960 dominant) {
        var blockTags = tagMap.getTagsByEntry(dominant);
        if (blockTags.isEmpty()) return null;

        var tagHolders = RAW_BLOCK_TAGS.get(blockTags.iterator().next().location());
        if (tagHolders == null) return null;

        return findDominantHolder(tagHolders, dominant);
    }

    @Nullable
    private static <T> class_6880<T> findDominantHolder(Collection<class_6880<T>> holders, class_2960 dominant) {
        for (var tagHolder : holders) {
            var holderKey = tagHolder.method_40230();
            if (holderKey.isPresent() && holderKey.get().method_29177().equals(dominant)) {
                return tagHolder;
            }
        }

        return null;
    }

    private static Set<class_2960> applyItemTags(UnifyConfig unifyConfig, TagMap<class_1792> globalTagMap, class_6880<class_1792> dominantItemHolder, Set<UnifyTag<class_1792>> dominantItemTags, class_2960 item) {
        var itemTags = globalTagMap.getTagsByEntry(item);
        Set<class_2960> changed = new HashSet<>();

        for (var itemTag : itemTags) {
            if (!unifyConfig.shouldInheritItemTag(itemTag, dominantItemTags)) continue;
            if (tryUpdatingRawTags(dominantItemHolder, itemTag, RAW_ITEM_TAGS)) {
                changed.add(itemTag.location());
            }
        }

        return changed;
    }

    private static Set<class_2960> applyBlockTags(UnifyConfig unifyConfig, TagMap<class_2248> blockTagMap, class_6880<class_2248> dominantBlockHolder, Set<UnifyTag<class_1792>> dominantItemTags, class_2960 item) {
        var blockTags = blockTagMap.getTagsByEntry(item);
        Set<class_2960> changed = new HashSet<>();

        for (var blockTag : blockTags) {
            if (!unifyConfig.shouldInheritBlockTag(blockTag, dominantItemTags)) continue;
            if (tryUpdatingRawTags(dominantBlockHolder, blockTag, RAW_BLOCK_TAGS)) {
                changed.add(blockTag.location());
            }
        }

        return changed;
    }

    private static <T> boolean tryUpdatingRawTags(class_6880<T> dominantHolder, UnifyTag<T> tag, Map<class_2960, Collection<class_6880<T>>> rawTags) {
        var tagHolders = rawTags.get(tag.location());
        if (tagHolders == null) return false;
        if (tagHolders.contains(dominantHolder)) return false; // already present, no need to add it again

        ImmutableSet.Builder<class_6880<T>> newHolders = ImmutableSet.builder();
        newHolders.addAll(tagHolders);
        newHolders.add(dominantHolder);

        rawTags.put(tag.location(), newHolders.build());
        return true;
    }

    private record TagRelation(class_2960 tag, class_2960 dominant, Set<class_2960> items) {}
}
