package com.almostreliable.unified.utils;

import com.almostreliable.unified.AlmostUnified;
import com.google.common.collect.*;
import javax.annotation.Nullable;
import net.minecraft.class_1792;
import net.minecraft.class_2960;
import net.minecraft.class_6880;
import java.util.Collection;
import java.util.Map;
import java.util.Set;

public class TagOwnerships {

    /**
     * A map holding relationships between reference tags and their owner tags.
     * <p>
     * Example:<br>
     * If the map contains the entry {@code minecraft:logs -> minecraft:planks},
     * any recipes where the tag {@code minecraft:logs} is being used, it will
     * replace the tag with {@code minecraft:planks}.
     * <p>
     * Map Key = Tag to replace<br>
     * Map Value = Tag to replace with
     */
    private final Map<UnifyTag<class_1792>, UnifyTag<class_1792>> refsToOwner;
    private final Multimap<UnifyTag<class_1792>, UnifyTag<class_1792>> ownerToRefs;

    /**
     * Creates a new TagOwnerships instance that contains immutable maps of all tag ownership relationships.
     * <p>
     * It is ensured that all owner tags are present in the {@code unifyTags} set, and that all reference tags
     * aren't present in the {@code unifyTags} set.
     *
     * @param unifyTags          The set of all unify tags in use.
     * @param tagOwnershipConfig The map of all tag ownership relationships.
     */
    public TagOwnerships(Set<UnifyTag<class_1792>> unifyTags, Map<class_2960, Set<class_2960>> tagOwnershipConfig) {
        ImmutableMap.Builder<UnifyTag<class_1792>, UnifyTag<class_1792>> refsToOwnerBuilder = ImmutableMap.builder();
        ImmutableMultimap.Builder<UnifyTag<class_1792>, UnifyTag<class_1792>> ownerToRefsBuilder = ImmutableMultimap.builder();

        tagOwnershipConfig.forEach((rawOwner, rawRefs) -> {
            for (class_2960 rawRef : rawRefs) {
                UnifyTag<class_1792> owner = UnifyTag.item(rawOwner);
                UnifyTag<class_1792> ref = UnifyTag.item(rawRef);

                if (!unifyTags.contains(owner)) {
                    AlmostUnified.LOG.warn(
                            "[TagOwnerships] Owner tag '#{}' is not present in the unify tag list!",
                            owner.location()
                    );
                    continue;
                }

                if (unifyTags.contains(ref)) {
                    AlmostUnified.LOG.warn(
                            "[TagOwnerships] Reference tag '#{}' of owner tag '#{}' is present in the unify tag list!",
                            ref.location(),
                            owner.location()
                    );
                    continue;
                }

                refsToOwnerBuilder.put(ref, owner);
                ownerToRefsBuilder.put(owner, ref);
            }
        });

        this.refsToOwner = refsToOwnerBuilder.build();
        this.ownerToRefs = ownerToRefsBuilder.build();
    }

    /**
     * Applies tag ownerships to the provided raw tags.
     * <p>
     * The raw tags are then processed by the game and actual tags are created.
     *
     * @param rawTags The raw tags to apply ownerships to.
     */
    public void applyOwnerships(Map<class_2960, Collection<class_6880<class_1792>>> rawTags) {
        Multimap<class_2960, class_2960> changedTags = HashMultimap.create();

        ownerToRefs.asMap().forEach((owner, refs) -> {
            var rawHolders = rawTags.get(owner.location());
            if (rawHolders == null) {
                AlmostUnified.LOG.warn("[TagOwnerships] Owner tag '#{}' does not exist!", owner.location());
                return;
            }

            ImmutableSet.Builder<class_6880<class_1792>> holders = ImmutableSet.builder();
            holders.addAll(rawHolders);
            boolean changed = false;

            for (UnifyTag<class_1792> ref : refs) {
                var refHolders = rawTags.get(ref.location());
                if (refHolders == null) {
                    AlmostUnified.LOG.warn(
                            "[TagOwnerships] Reference tag '#{}' of owner tag '#{}' does not exist!",
                            ref.location(),
                            owner.location()
                    );
                    continue;
                }

                for (class_6880<class_1792> holder : refHolders) {
                    holders.add(holder);
                    holder.method_40230().ifPresent(key -> changedTags.put(owner.location(), key.method_29177()));
                    changed = true;
                }
            }

            if (changed) {
                rawTags.put(owner.location(), holders.build());
            }
        });

        if (!changedTags.isEmpty()) {
            changedTags.asMap().forEach((tag, items) -> {
                AlmostUnified.LOG.info("[TagOwnerships] Modified tag '#{}', added {}", tag, items);
            });
        }
    }

    /**
     * Gets the owner tag for the provided reference tag.
     *
     * @param tag The reference tag to get the owner for.
     * @return The owner tag, or null if the provided tag is not a reference tag.
     */
    @Nullable
    public UnifyTag<class_1792> getOwnerByTag(UnifyTag<class_1792> tag) {
        return refsToOwner.get(tag);
    }

    /**
     * Gets all reference tags for the provided owner tag.
     *
     * @param tag The owner tag to get the references for.
     * @return A collection of all reference tags.
     */
    public Collection<UnifyTag<class_1792>> getRefsByOwner(UnifyTag<class_1792> tag) {
        return ownerToRefs.get(tag);
    }
}
